package com.aimms.aimmssdk.examples.views;

import com.aimms.aimmssdk.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** 
 * This example demonstrates views on data, initializing set data in a single call and dense data input.  
 * The combination of the first and last one can be powerful, since the dense input functionality modifies the entire domain
 * and the views allow for flexible use of domain restrictions. 
 */
class Views {

    ISession m_Session;

    public Views(ISession session) {
        m_Session = session;
    }

    void createViews() {
        // It is possible to create a view on data by placing restrictions on the domains.
        // This is a three step proces:
        IMultiDimData unitTransportCost = m_Session.openMultiDim("unitTransportCost");

        // 1: Create an IFilter
        IFilter filter = unitTransportCost.createFilter();

        // 2: Add domain restrictions to the filter
        filter.restrict(0, "Oslo");   // Restricts the 0-th dimension to the element value "Oslo".

        // 3: Create an IDataView 
        IDataView utcFromOslo = unitTransportCost.openView(filter);

        // This view has a lower dimension than the unitTransportCost, since the data is fixed on the first level. 
        int dim = utcFromOslo.getDimension();
        assert (dim == 1);

        utcFromOslo.close();


        // It is possible to add restrictions on other levels or to change earlier restrictions  
        filter.restrict(1, "Berlin");
        filter.restrict(0, "London");
        IDataView utcFromLondonToBerlin = unitTransportCost.openView(filter);

        // If the view has become scalar ( all levels are fixed), 
        // the IDataView can be cast to an IScalarDataView for some additional user friendly methods.
        IScalarDataView scalarview = (IScalarDataView) utcFromLondonToBerlin;
        double utcFromLondonToBerlin_Value = scalarview.asDouble();

        utcFromLondonToBerlin.close();

        filter.close();
        unitTransportCost.close();
    }
    
    
    void setLabels() {
        // Assuming set data is stored in arrays, it can be easily send to the Aimms model by using setLabels.
        String[] depots = {"London", "Oslo"};
        String[] customers = {"Berlin", "Praag", "Oslo", "Rome"};

        ISetData set;
        set = m_Session.openSet("Depots");
        set.setLabels(depots);
        set.close();
        set = m_Session.openSet("Customers");
        set.setLabels(customers);
        set.close();

        // It is also possible to increase the set using appendLabels .
        String[] extradepots = {"London", "Madrid"};
        set = m_Session.openSet("Depots");
        set.appendLabels(extradepots);
        // The set depots is now {"London","Oslo","Madrid"}.
    }

    void setValues() {
        // Multidimensional data can be send to the Aimms Model using setValues.
        // The length of the array must be supplied; and must be the product of the cardinality of the domains.
        double [] supplyValues = {3.0, 5.0, 4.0};
        double [] demandValues = {2.7, 3.2, 3.0, 2.9};

        IMultiDimData supply;
        supply = m_Session.openMultiDim("Supply");
        supply.setValues(supplyValues);
        supply.close();

        IMultiDimData demand = m_Session.openMultiDim("Demand");
        demand.setValues(demandValues);
        demand.close();

        // note: Higher dimensional data still uses a 1 dimensional array.
        double[] unitTransportValues = {
            23.7, 45.0, 23.2, 45.3,
            36.7, 38.8, 21.4, 12.6,
            66.2, 22.0, 37.1, 13.8};

        IMultiDimData unitTransportCost = m_Session.openMultiDim("unitTransportCost");
        unitTransportCost.setValues(unitTransportValues);
        unitTransportCost.close();
    }

    

    void assignToViews() {
        // Assigning data to a view has two (potential) advantages:
        //  - reduction of the dimension 
        //  - sparse use of the dense function setValues

        // create a view
        IMultiDimData unitTransportCost = m_Session.openMultiDim("unitTransportCost");
        IFilter filter = unitTransportCost.createFilter();
        filter.restrict(0, "Oslo");
        IDataView utcFromOslo = unitTransportCost.openView(filter);
        // notice that utcFromOslo is only 1-dimensional

        // use setValue
        utcFromOslo.setValue(new Tuple("Berlin"), 34.7);

        // use setValues
        double[] utcfromhilversum = {32.8, 24.3, 46.2, 12.0};
        utcFromOslo.setValues(utcfromhilversum);

        filter.close();
        unitTransportCost.close(); // Closing the IMultiDimData closes all the views as well.

    }

    void readFiltered() {
        // The IDataView class allow the use of iterators, but the filtered iterator can also be created directly on the IMultiDimData.

        IMultiDimData unitTransportCost = m_Session.openMultiDim("unitTransportCost");

        // Construct a filter for the costs from Oslo.
        IFilter filter = unitTransportCost.createFilter();
        filter.restrict(0, "Oslo");

        // create a view...
        IDataView utcFromOslo = unitTransportCost.openView(filter);
        // ... and an iterator on it
        IIterator it1 = utcFromOslo.createIterator();
        // ... or create an iterator with the filter
        IIterator it2 = unitTransportCost.createIterator(filter);

        it1.close();
        it2.close();
        utcFromOslo.close();
        unitTransportCost.close();
    }
}
