package com.aimms.aimmssdk.examples.procedureargs;

import com.aimms.aimmssdk.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** 
 * This example demonstrates the use of a procedure with arguments.
 * Procedure arguments can be supplied in two ways:
 * - Locals can be retrieved using getArgument. This yields an IData object
 *   which can be cast to the actual class and can be used to read/write data directly into 
 *   those locals.
 * - Global arguments can be used.
 */
class ProcedureArgs {

    static private Logger logger = LoggerFactory.getLogger(ProcedureArgs.class);

    static void run(ISession session) {
        // retrieve the procedure
        IProcedure maxCheck = session.openProcedure("MaxCheck");

        // MaxCheck verifies that all values of P1(argument0) are less or equal to MaxValue(argument1).
        // If so, it returns 1, else it returns 0.

        // the formal signature of MaxCheck:
        // argument0: input;   double parameter over FilledSet
        // argument1: input;   double parameter Scalar 

        // Since we do not have scalar to supply as second argument, we use the argument of the procedure directly:
        IData arg = maxCheck.getArgument(1);
        assert (arg.getType() == DataType.tScalar);
        assert (arg.getValueType() == ValueType.vtDouble);
        IScalarData maxValue = (IScalarData) (arg);
        maxValue.setValue(50.0);
        
        // We also need to open the parameter to be checked.
        IMultiDimData testparameter = session.openMultiDim("FilledParameter");

        // Determine whether filledparameter has no value over 50:
        int result = maxCheck.run(testparameter, maxValue);
        
        // Since the maximum  is 9 the result will  be true:
        logger.info("result={}; expected 1", result);
        assert (result == 1);

        // Determine whether filledparameter does have a value over 5: 
        maxValue.setValue(5.0);
        result = maxCheck.run(testparameter, maxValue);
        
        //  Since the maximum  is 9 the result will be false:
        logger.info("result={}; expected 0", result);
        assert (result == 0);

        testparameter.close();

        // A procedure will close its arguments, so MaxValue does not need to be closed, although it is allowed.
        maxValue.close();
        maxCheck.close();
    }
}
