package com.aimms.aimmssdk.examples.performance;

import com.aimms.aimmssdk.*;
import java.util.Date;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This example demonstrates the performance difference between three ways to
 * assign data to a set and to assign a 3 dimensional parameter on 50% of its
 * domain positions.
 */
class Performance {

    static private Logger logger = LoggerFactory.getLogger(Performance.class);
    String m_Location;
    String m_Project;
    String[] m_Labels;
    ISession m_Aimms = null;
    ISetData m_SubSet = null;
    IMultiDimData m_LargePar = null;
    long m_StartTime = 0;

    public Performance(String aimmsLocation, String project, String[] labels) {
        m_Location = aimmsLocation;
        m_Project = project;
        m_Labels = labels;
    }

    /**
     * The convenient way: Use a Tuple object constructed with labels to insert
     * each data entry. This is inefficient due to the required string
     * comparisons.
     */
    public void convenientAssignLabels() {

        openSessionAndData();
        int setsize = m_Labels.length;

        start("Assign string labels using insert and the Tuple class");

        double value = 0.0;

        for (int i = 0; i < setsize; i += 2) {  // a density  of 50%, skip all odd positions of this dimension
            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    ++value;
                    m_LargePar.insert(new Tuple(m_Labels[i], m_Labels[j], m_Labels[k]), value);
                }
            }
        }
        m_Aimms.updateData();

        stop();

        closeDataAndSession();
    }

    /**
     * Ordinals, ITuple and setValue: The more efficient way: Use ordinals to
     * construct an ITuple and add the datapoint with it. Since the SDK uses the
     * 'natural' Aimms ordering, the ordering of the data is not guaranteed to
     * be the order in which the set data is supplied. So use ordinals with
     * care. See also the section on "Set element ordering" in the Execution
     * Efficiency Cookbook chapter in the Aimms Language Reference.
     */
    public void efficientAssignLabels() {

        openSessionAndData();
        int setsize = m_Labels.length;

        start("Assign string labels first to the set and use the ITuple interface, ordinals and setValue to assign the values on the parameter");


        for (int i = 0; i < setsize; ++i) {
            m_SubSet.add(m_Labels[i]);
        }

        m_Aimms.updateData(); // Update the set data to make label lookup faster.

        ITuple tup = m_LargePar.createTuple();
        double value = 0.0;
        try {
            for (int i = 1; i <= setsize; i += 2) {    // A density  of 50%: skip al odd positions on this dimension.

                // Set each dimension only when it changes. 
                tup.getElement(0).setOrdinal(i);
                for (int j = 1; j <= setsize; ++j) {
                    tup.getElement(1).setOrdinal(j);
                    for (int k = 1; k <= setsize; ++k) {
                        tup.getElement(2).setOrdinal(k);
                        ++value;

                        m_LargePar.setValue(tup, value);
                    }
                }
            }

            m_Aimms.updateData();
        } catch (AimmsRuntimeException ex) {
            // setValue throws an AimmsRuntimeException if the elements of the tuple are not present in the index domain.
            logger.error("Catched a runtime exception: {}", ex.getMessage());
            m_Aimms.clearBuffers();

        } finally {

            stop();
            closeDataAndSession();
        }


    }

    /**
     * setValues: Efficient and convenient if the data to assign is dense. The
     * setLabels (for sets) and setValues (for data) methods do allow for dense
     * allocation of an identifier. The data will be added in the natural
     * ordering of the domains, thus combining ease of use with efficiency.
     */
    public void denseAssignLabels() {

        openSessionAndData();
        int setsize = m_Labels.length;
        logger.info("setsize: {} ", setsize);
        start("Assign string labels first to the set using setLabels and then use setValues on the IMultiDimData to assign the values to the parameter");


        m_SubSet.setLabels(m_Labels);

        // UpdateData is not neccesary here because after the setLabels method the set is immediately up-to-date.

        double[] values = new double[setsize * setsize * setsize];

        // Whether a loop like this is neccesary depends on the source of the values.
        double value = 0;
        int pos = 0;
        for (int i = 0; i < setsize; i += 2) {

            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    values[pos++] = ++value;
                }
            }
            // Now insert a two dimensional slice with zero's (the default value), because this is a dense assign method but the data is not fully dense.
            for (int z = 0; z < setsize * setsize; ++z) {
                values[pos++] = 0;
            }
        }

        m_LargePar.setValues(values);

        m_Aimms.updateData();

        stop();

        closeDataAndSession();
    }

    /**
     * Slices: Efficient if the data to be assigned is sparse and distributed in
     * a way slices can be defined. In this example multiple views are created
     * that are all restricted on the same level on a single element. This
     * reduces the dimension.
     */
    public void sliceAssignLabels() {

        openSessionAndData();
        int setsize = m_Labels.length;

        start("Assign string labels first to the set using setLabels and then use setValues on IDataView slices of the IMultiDimData to assign the values to the parameter");


        m_SubSet.setLabels(m_Labels);


        double[] values = new double[setsize * setsize];

        // Create a filter to slice the parameter.
        IFilter filter = m_LargePar.createFilter();

        double value = 0;
        for (int i = 0; i < setsize; i += 2) {
            // For each position of this dimension to which we will assign data, construct the filter to create a DataView. 
            // Notice the +=2, we only need views on the even positions.  
            filter.restrict(0, m_Labels[i]);
            IDataView slice = m_LargePar.openView(filter);

            int pos = 0;
            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    values[pos++] = ++value;
                }
            }
            slice.setValues(values);
        }

        m_Aimms.updateData();

        stop();

        closeDataAndSession();
    }

    /**
     * Filtering by call domain: Efficient if the data is sparse and an
     * appropriate subset is available to filter with.
     */
    public void filteredAssignLabels() {

        openSessionAndData();
        int setsize = m_Labels.length;

        start("Assign string labels first to the set using setLabels and use setValues on an IDataView filtered by a subset to assign the values on the parameter");

        m_SubSet.setLabels(m_Labels);

        // Open a subset on the domainset to use as a call domain on the level to restrict.
        ISetData subsubset = m_Aimms.openSet("SubSubSet");
        IElement element = m_SubSet.createElement();

        for (int o = 1; o <= setsize; o += 2) {
            element.setOrdinal(o);
            subsubset.add(element);
        }
        m_Aimms.updateData();

        double[] values = new double[setsize / 2 * setsize * setsize];

        // Create a filter to slice the parameter.
        IFilter filter = m_LargePar.createFilter();
        filter.restrict(0, subsubset);
        IDataView view = m_LargePar.openView(filter);


        double value = 0;
        int pos = 0;
        for (int i = 0; i < setsize; i += 2) {
            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    values[pos++] = ++value;
                }
            }
        }
        view.setValues(values);

        m_Aimms.updateData();

        stop();

        subsubset.close();
        closeDataAndSession();


    }

    // Helper functions
    private void openSessionAndData() {
        m_Aimms = null;
        m_Aimms = AIMMS.openSession(m_Location, m_Project);
        if (m_Aimms == null) {
            return;
        }
        m_SubSet = m_Aimms.openSet("LargeSubSet");
        m_LargePar = m_Aimms.openMultiDim("LargePar1");
    }

    private void closeDataAndSession() {
        m_LargePar.close();
        m_SubSet.close();
        m_Aimms.close();
    }

    private void start(String testDescription) {
        logger.info("Start Running {}...", testDescription);
        m_StartTime = new Date().getTime();
    }

    private void stop() {
        long stop = new Date().getTime();
        long dif = stop - m_StartTime;
        logger.info("This took {} milliseconds", dif);
        logger.info("");
    }
}
