package com.aimms.aimmssdk.examples.callback;

import com.aimms.aimmssdk.*;
import java.util.ArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This example demonstrates the custom handling of errors. It overrides the
 * default behavior in which AimmsExceptions are thrown in all error cases in
 * two ways: The LoggingCallback logs all minor errors, and keeps throwing the
 * severe ones The DebuggingCallback logs as well, and asserts where errors can
 * only be expected if the model was called erroneously.
 */
class Callback {

    static private Logger logger = LoggerFactory.getLogger(Callback.class);

    static class LoggingCallback implements ICallBack {

        public void onMessages(ArrayList<AimmsMessage> messages) {
            // Overridden behavior:
            // Only throw if an error is caused by the communication layer or the AIMMS model. 
            // Log the message in other cases. 

            for (int m = 0; m < messages.size(); ++m) {
                AimmsMessage msg = messages.get(m);
                switch (msg.code) {
                    case AM_Info:
                        logger.info(msg.message);
                        break;
                    case AM_Warning:
                        logger.warn(msg.message);
                        break;
                    case AM_InvalidUse:
                        logger.error("InvalidUse: {}", msg.message);
                        break;
                    case AM_Runtime:
                        logger.error("Runtime: {}", msg.message);
                        break;
                    case AM_License:
                        throw new AimmsLicenseException(msg.message);
                    case AM_Communication:
                        throw new AimmsCommunicationException(msg.message);
                    case AM_General:  // AIMMS Model errors
                        throw new AimmsException(msg.code, msg.message);
                    default:
                        throw new AimmsException(msg.code, msg.message);
                }

                // The rationale behind this distinction is that in the cases in which exceptions are thrown now, 
                // the state of the data is not known. For example: if during the flush of the buffers an error on 
                // a specific value is detected by the AIMMS model, it is not known which part of the remaining data 
                // is already written to the model. 
            }
        }
    }

    static class DebuggingCallback implements ICallBack {


    public void onMessages(ArrayList<AimmsMessage> messages) {
        // Overridden behavior:
        // 
        //  assert whether an AM_InvalidUse is encountered, because that is an indication of a bug. 
        //
        // For example: adding a string to a numerical parameter gives AM_InvalidUse. 


        for (int m = 0; m < messages.size(); ++m) {
            AimmsMessage msg = messages.get(m);

            switch (msg.code) {
                case AM_InvalidUse:
                    logger.error(msg.message);
                    assert (false);
                    break;
                default:
                    logger.error("{}:{}", msg.code, msg.message);
                    break;
            }

        }
    }
}
    
    
    
    public static void run(ISession session) {
        LoggingCallback cb = new LoggingCallback();
        //Attach the custom Callback to the session.
        session.setCallBack(cb);

        //To illustrate the behavior of our callback, make some mistakes:

        //Try to retrieve the set Locations as a multidimensional parameter.
        //This is an InvalidUse error: 
        //   LoggingCallback will not throw
        //   DebuggingCallback will assert here. 
        IMultiDimData parameterLocations = session.openMultiDim("Locations");

        if (parameterLocations == null) {
            logger.error("Expected an InvalidUse error!");
        }

        IMultiDimData parameterSupply = session.openMultiDim("Supply");
        //Do not forget to check the result if the callback does not throw on each error!
        if (parameterSupply == null) {
            logger.error("Did not expect an error here!");
        } else {
            // Since we did not add any depots, we will get a runtime error here:
            parameterSupply.setValue(new Tuple("London"), 20.0);

            // Using "insert" will resolve this error, since it adds London to the index domain of Supply if not existing. 
            parameterSupply.insert(new Tuple("London"), 20.0);

            // Supply is a numeric parameter: it will not allow strings as value:
            // DebuggingCallback will assert.
            parameterSupply.insert(new Tuple("Paris"), "thirty");
        }

        IMultiDimData parameterDemand = session.openMultiDim("Demand");

        if (parameterDemand == null) {
            logger.error("Did not expect an error here!");
        } else {
            ITuple tup = parameterDemand.createTuple();

            // "London" exists in the set Locations (because we added it to Depots by inserting in Supply),
            // but the index domain of Demand is Customers, which  is still empty.
            // The nocheck_ functions are less expensive because they do not check this in advance.
            // Cout will not show an error after these actions:
            tup.getElement(0).nocheck_setLabel("London");
            parameterDemand.setValue(tup, 90);
            tup.close();

            // But this performance improvement comes with a downside:
            try {
                // The mistake will become apparent when the data is sent to the model. 
                // Then it is undetermined wether the inserted value for "London" in Supply is indeed inserted in Aimms.
                // If Londen is added to the domainset and this set is flushed before the Supply is flushed,
                // the Supply of Londen will be set to its new value without an error.
                // If Londen is not added or Supply is flushed earlier (which may be due to an internal flush),
                // then an error will occur and even other buffered modifications to Supply may not have been inserted in Aimms.
                session.updateData();

            } catch (AimmsException e) {
                logger.error(e.getMessage());
            }

            parameterSupply.close();
            parameterDemand.close();
        }
    }
}
