#include <iostream>
#include <time.h>
#include <assert.h>
#include "aimms/Include.h"

// This example demonstrates views on data, initializing set data in a single call and dense data input.  
// The combination of the first and last one can be powerful, since the dense input functionality modifies the entire domain
// and the views allow for flexible use of domain restrictions. 

void createViews(aimms::ISession* session)
{
    // It is possible to create a view on data by placing restrictions on the domains.
    // This is a three step proces:
    aimms::IMultiDimData* unitTransportCost = session->openMultiDim("unitTransportCost");

    // 1: Create an IFilter
    aimms::IFilter* filter = unitTransportCost->createFilter(); 

    // 2: Add domain restrictions to the filter
    filter->restrict(0,"Oslo");   // Restricts the 0-th dimension to the element value "Oslo". 

    // 3: Create an IDataView 
    aimms::IDataView* utcFromOslo = unitTransportCost->openView(filter);

    // This view has a lower dimension than the unitTransportCost, since the data is fixed on the first level. 
    int dim = utcFromOslo->getDimension();
    assert(dim == 1);

    utcFromOslo->close();

    // It is possible to add restrictions on other levels or to change earlier restrictions.  
    filter->restrict(1,"Berlin");   
    filter->restrict(0,"London");  
    aimms::IDataView* utcFromLondonToBerlin = unitTransportCost->openView(filter);

    // If the view has become scalar (all levels are fixed), 
    // the IDataView can be cast to an IScalarDataView for some additional user friendly methods.
    aimms::IScalarDataView* scalarview = dynamic_cast<aimms::IScalarDataView*>(utcFromLondonToBerlin);
    double utcFromLondonToBerlin_Value = scalarview->asDouble();

    utcFromLondonToBerlin->close();

    filter->close();
    unitTransportCost->close();
}



void setLabels(aimms::ISession* session)
{
    // Assuming set data is stored in arrays, it can be easily sent to the Aimms model by using setLabels.
    const char* depots[] = {"London","Oslo"};
    const char* customers[] = {"Berlin","Prague","Oslo", "Rome"};

    aimms::ISetData* set;
    set = session->openSet("Depots");
    set->setLabels(depots,2);
    set->close();
    set = session->openSet("Customers");
    set->setLabels(customers,4);
    set->close();

    // It is also possible to increase the set using appendLabels.
    const char* extradepots[] = {"London","Madrid"};
    set = session->openSet("Depots");
    set->appendLabels(extradepots,2);
    // The set "Depots" is now {'London','Oslo','Madrid'}.
}

void setValues(aimms::ISession* session)
{
    // Multidimensional data can be sent to the Aimms Model using setValues.
    // The length of the array must be supplied; and must be the product of the cardinality of the domains.
    const double supplyValues[] = {3.0, 5.0, 4.0};
    const double demandValues[] = {2.7, 3.2, 3.0, 2.9};

    aimms::IMultiDimData* supply;
    supply = session->openMultiDim("Supply");
    supply->setValues(supplyValues,3);
    supply->close();

    aimms::IMultiDimData* demand = session->openMultiDim("Demand");
    demand->setValues(demandValues,4);
    demand->close();

    // note: Higher dimensional data still uses a 1 dimensional array.
    const double unitTransportValues[] = 
    {23.7, 45.0, 23.2, 45.3, 
    36.7, 38.8, 21.4, 12.6, 
    66.2, 22.0, 37.1, 13.8};

    aimms::IMultiDimData* unitTransportCost = session->openMultiDim("unitTransportCost");
    unitTransportCost->setValues(unitTransportValues,4*3);
    unitTransportCost->close();
}



void assignToViews(aimms::ISession* session)
{
    // Assigning data to a view has two (potential) advantages:
    //  - reduction of the dimension 
    //  - sparse use of the dense function setValues

    // create a view
    aimms::IMultiDimData* unitTransportCost = session->openMultiDim("unitTransportCost");
    aimms::IFilter* filter = unitTransportCost->createFilter(); 
    filter->restrict(0,"Oslo");  
    aimms::IDataView* utcFromOslo = unitTransportCost->openView(filter);
    // notice that utcFromOslo is only 1-dimensional

    // use setValue
    utcFromOslo->setValue(aimms::Tuple("Berlin"), 34.7);

    // use setValues
    double utcfromhilversum[] = {32.8,24.3,46.2,12.0};
    utcFromOslo->setValues(utcfromhilversum, 4);

    filter->close();
    unitTransportCost->close(); // Closing the IMultiDimData closes all the views as well.

}

void readFiltered(aimms::ISession* session)
{
    // The IDataView class allows the use of iterators, but the filtered iterator can also be created directly on the IMultiDimData.

    aimms::IMultiDimData* unitTransportCost = session->openMultiDim("unitTransportCost");

    // Construct a filter for the costs from Oslo.
    aimms::IFilter* filter = unitTransportCost->createFilter(); 
    filter->restrict(0,"Oslo");  

    // create a view...
    aimms::IDataView* utcFromOslo = unitTransportCost->openView(filter);
    // ... and an iterator on it
    aimms::IIterator* it1 = utcFromOslo->createIterator();
    //... or create an iterator with the filter
    aimms::IIterator* it2 = unitTransportCost->createIterator(filter);

    it1->close();
    it2->close();
    utcFromOslo->close();
    unitTransportCost->close();
}

int main(int argc, const char* argv[]) 
{
    if (argc != 3) {
        std::cerr << "Invalid number of arguments. usage: <location of AIMMS>  <location of project>" << std::endl; 
        return 1;
    }  

    aimms::ISession* session = 0;
    try {
        // the examples
        session = aimms::openSession(argv[1],argv[2]); 

        setLabels(session);
        setValues(session);
        createViews(session);
        assignToViews(session);
        readFiltered(session);

    } catch (std::exception& e){

        std::cerr << e.what();

        if (session) {
            session->close(); 
        }
        return 1;
    }

    try{
        session->close();
        return 0;
    }catch (std::exception& e){
        std::cerr << e.what();
        return 1;
    }
}





