#include "aimms/Include.h"
#include <assert.h>

// This example demonstrates the use of a procedure with arguments.
// Procedure arguments can be supplied in two ways:
// - Locals can be retrieved using getArgument. This yields an aimms::IData object
//   which can be cast to the actual class and can be used to read/write data directly into 
//   those locals.
// - Global arguments can be used.

int main(int argc, const char* argv[]) 
{
    if (argc != 3) {
        std::cerr << "Invalid number of arguments. usage: <location of AIMMS>  <location of project>" << std::endl; 
        return 1;
    }  

    aimms::ISession* session = 0;

    try{
        session = aimms::openSession(argv[1],argv[2]);

        // Retrieve the procedure.
        aimms::IProcedure* maxCheck = session->openProcedure("MaxCheck");
        
        // MaxCheck verifies that all values of P1(argument0) are less than or equal to MaxValue(argument1).
        // If so, it returns 1, else it returns 0.

        // the formal signature of MaxCheck:
        // argument0: input;   double parameter over FilledSet
        // argument1: input;   double parameter Scalar 
 
        
        // Since we do not have a scalar to supply as second argument, we use the argument of the procedure directly:
        aimms::IData* arg = maxCheck->getArgument(1);
        assert(arg->getType() == aimms::tScalar);
        assert(arg->getValueType() == aimms::vtDouble);
        aimms::IScalarData* maxValue = dynamic_cast<aimms::IScalarData*>(arg);
        maxValue->setValue(50.0);

        // We also need to open the parameter to be checked.
        aimms::IMultiDimData* testParameter = session->openMultiDim("FilledParameter");
    
        // Determine whether filledparameter has no value over 50:
        int result = maxCheck->run(testParameter, maxValue);

        // Since the maximum  is 9 the result will be true:
        assert(result == 1);

        // Determine whether filledparameter does have a value over 5: 
        maxValue->setValue(5.0);   
        result = maxCheck->run(testParameter, maxValue);

        // Since the maximum  is 9 the result will be false:
        assert(result == 0); 

        testParameter->close();

        // A procedure will close its arguments, so MaxValue does not need to be closed, although it is allowed.
        maxValue->close();
        maxCheck->close();

    }  catch (std::exception& e){
        std::cerr << e.what();

        if (session) {
            session->clearBuffers(); // discards all buffered (and since there is an exception caught, maybe errorneous) modifications 
            session->close();
        }
        return 1;
    }

    try{
        session->close();
        return 0;
    } catch (std::exception& e){
        std::cerr << e.what();
        return 1;
    }
}