#include "aimms/Include.h"
#include <iostream>
#include <assert.h>

// This example demonstrates some easily made mistakes using the SDK.

// Using set ordering. 
// The SDK uses only the 'natural' Aimms ordering, which is the creation order of the labels.
// See also the section on "Set element ordering" in the Execution Efficiency Cookbook chapter in the Aimms Language Reference.
void useOrdinals(aimms::ISession* session)
{
    aimms::ISetData* setCustomers = session->openSet("Customers");
    setCustomers->add("Oslo"); 
    setCustomers->add("Madrid"); 

    // If the root set was completely empty until now, the order will be as expected.
    aimms::ISetIterator* itCustomers = setCustomers->createIterator();
    itCustomers->next();
    assert( std::string("Oslo") == itCustomers->element()->getLabel());
    itCustomers->next();
    assert( std::string("Madrid") == itCustomers->element()->getLabel());


    aimms::ISetData* setDepots = session->openSet("Depots");
    setDepots->add("London"); 
    setDepots->add("Oslo"); 

    // Since Depots and Customers share a rootset (Locations), 
    // "Oslo" was added to the rootset before "London".
    // It will be the first Depot!
    aimms::ISetIterator* itDepots = setDepots->createIterator();
    itDepots->next();
    assert( std::string("Oslo") == itDepots->element()->getLabel());
    itDepots->next();
    assert( std::string("London") == itDepots->element()->getLabel());
}

// Writing sets.
// Set write actions are recursive:
// - adding a label to a subset will also add it to the superset
// - deleting a label from a set will also delete it from its subsets, 
// - emptying a set will empty all its subsets.
// - setLabels() has an implicit empty, therefore setLabels empties all the subsets
// 
void initializeSets(aimms::ISession* session) 
{
    // the data:
    const char* locations[] = {"London", "Oslo" , "Madrid" };
    const char* customers[] = {"Oslo", "Madrid" };
    const char* depots[] = {"London", "Oslo" };

    // adding the data incorrectly:
    {
        aimms::ISetData* setCustomers = session->openSet("Customers");
        setCustomers->setLabels(customers,2); // This will also add Oslo and Madrid to the superset Locations.
        aimms::ISetData* setDepots = session->openSet("Depots");
        setDepots->setLabels(depots,2); // This will also add London to the superset Locations, the others were already present.
        aimms::ISetData* setLocations = session->openSet("Locations");
        setLocations->setLabels(locations,3); // This will empty the set and its subsets, and then adds London and Oslo to only itself.

        assert(setDepots->getCardinality() == 0); // Probably not intended.
    }
}


int main(int argc, const char* argv[]) 
{
    if (argc != 3) {
        std::cerr << "Invalid number of arguments. usage: <location of AIMMS>  <location of project>" << std::endl; 
        return 1;
    }  

    aimms::ISession* session = 0;

    try {

        // the examples
        session = aimms::openSession(argv[1],argv[2]); 

        useOrdinals(session);
        initializeSets(session);

    } catch (std::exception& e){

        std::cerr << e.what();

        if (session) {
            session->close(); 
        }
        return 1;
    }

    try{
        session->close();
        return 0;
    } catch (std::exception& e){
        std::cerr << e.what();
        return 1;
    }
} 
