#include "aimms/Include.h"
#include <iostream>
#include <time.h>
#include <assert.h>


// This example demonstrates the performance difference between three ways to
// assign data to a set and to assign a 3 dimensional parameter on 50% of its
// domain positions.   


// the data used to fill the domainsets in this example
const char* s_StringLabels[325] =  {"ab", "ac", "ad", "ae", "ah", "ai", "al", "am", "an", "ar", "as", "at", "aw", "ax", "ay", "aah", "aam", "abb", "abc", "abo", "abs", "abt", "aby", "ace", "ach", "act", "add", "adj", "ado", "ads", "adv", "adz", "afb", "aft", "aga", "age", "ago", "aha", "ahs", "ahu", "aid", "ail", "aim", "air", "ais", "ait", "ake", "ala", "alb", "ale", "all", "alp", "als", "alt", "ama", "amp", "amt", "amu", "amy", "ana", "and", "ani", "ann", "ano", "ant", "any", "ape", "apo", "app", "apt", "ara", "arc", "are", "arf", "ark", "arm", "ars", "art", "asa", "ash", "ask", "asp", "ass", "ate", "aud", "auf", "auk", "aum", "aux", "ava", "ave", "avg", "awe", "awk", "awl", "awm", "awn", "axe", "aye", "aahs", "abay", "abba", "abbe", "abbr", "abed", "abet", "abib", "abid", "abit", "able", "ably", "abox", "abra", "abut", "abye", "acct", "aced", "aces", "ache", "achy", "acid", "aclu", "acme", "acne", "acre", "acts", "acyl", "adam", "adar", "adaw", "adds", "adit", "ados", "adry", "advt", "adze", "aeon", "aero", "aery", "afar", "afer", "affy", "afro", "agar", "aged", "agen", "ages", "agha", "agin", "agio", "agni", "agog", "agon", "agre", "ague", "ahem", "ahey", "ahoy", "aide", "aids", "aiel", "ails", "aims", "aino", "airs", "airy", "ajar", "ajog", "akin", "alae", "alai", "alan", "alar", "alas", "alba", "albe", "albs", "alco", "alee", "alem", "ales", "alew", "alfa", "alga", "alii", "alit", "alls", "ally", "alma", "alme", "alms", "aloe", "alow", "alps", "also", "alto", "alum", "amah", "ambo", "amel", "amen", "amex", "amia", "amic", "amid", "amir", "amis", "amit", "amma", "ammo", "amok", "amps", "amts", "amyl", "anal", "anan", "anas", "ands", "anes", "anet", "anew", "anil", "anis", "ankh", "anna", "anne", "anno", "anoa", "anon", "ansa", "ansi", "anta", "ante", "anti", "ants", "anus", "apar", "aped", "aper", "apes", "apex", "apis", "apod", "apse", "apus", "aqua", "arab", "arak", "arch", "arco", "arcs", "area", "ares", "aret", "arew", "argo", "aria", "arid", "aril", "arks", "arms", "army", "arna", "arow", "arse", "arts", "arty", "arum", "asap", "asci", "asea", "ashy", "asia", "asks", "asps", "asse", "assn", "asst", "atma", "atmo", "atom", "atop", "atte", "attn", "atty", "atwo", "aube", "auks", "auld", "auln", "aune", "aunt", "aura", "auto", "avdp", "avel", "aver", "aves", "avid", "avie", "avis", "avow", "away", "awed", "awes", "awls", "awns", "awny", "awol", "awry", "axal", "axed", "axel", "axes", "axil", "axis", "axle", "axon", "ayah", "ayen", "ayes", "ayle", "ayme", "ayry", "azym"};

class Performance{

public:
    Performance(const char* aimmsLocation, const char* project) 
        : m_Location(aimmsLocation)
        , m_Project(project)
        , m_Aimms(0)
        , m_SubSet(0)
        , m_LargePar(0)
        , m_StartTime(0)
    {
    }


public:
    /**
    * The convenient way: Use a Tuple object constructed with labels to insert
    * each data entry. This is inefficient due to the required string
    * comparisons.
    */
    void convenientAssignLabels(int setsize) {

        openSessionAndData();

        start("Assign string labels using insert and the Tuple class");

        double value = 0.0;

        for (int i = 0; i < setsize; i += 2) {  // a density  of 50%, skip al odd positions on this dimension
            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    ++value;
                    m_LargePar->insert(aimms::Tuple(s_StringLabels[i], s_StringLabels[j], s_StringLabels[k]), value);
                }
            }
        }
        m_Aimms->updateData();

        stop();

        closeDataAndSession();
    }

    /*
    * Ordinals, ITuple and setValue: The more efficient way: Use ordinals to
    * construct an ITuple and add the datapoint with it. Since the SDK uses the
    * 'natural' Aimms ordering, the ordering of the data is not guaranteed to
    * be the order in which the set data is supplied. So use ordinals with
    * care. See also the section on "Set element ordering" in the Execution
    * Efficiency Cookbook chapter in the Aimms Language Reference.
    */
    void efficientAssignLabels(int setsize) {

        openSessionAndData();

        start("Assign string labels first to the set and use the ITuple interface, ordinals and setValue to assign the values on the parameter");


        for (int i = 0; i < setsize; ++i) {
            m_SubSet->add(s_StringLabels[i]);
        }

        m_Aimms->updateData(); // Update the set data to make label lookup faster.

        aimms::ITuple* tup = m_LargePar->createTuple();

        double value = 0.0;
        try{
            for (int i = 1; i <= setsize; i += 2) {    // A density  of 50%: skip all odd positions of this dimension.

                // Set each dimension only when it changes. 
                tup->getElement(0)->setOrdinal(i);      // Here the getElement method is used to get access to the indivual elements per dimension
                // The operator [] is defined on ITuple as well. For hints on how to use these, see the language reference.
                for (int j = 1; j <= setsize; ++j) {    
                    tup->getElement(1)->setOrdinal(j);
                    for (int k = 1; k <= setsize; ++k) {
                        tup->getElement(2)->setOrdinal(k);
                        ++value;
                        m_LargePar->setValue(tup, value);
                    }
                }
            }
            m_Aimms->updateData();
        } catch( aimms::RuntimeException& ex){
            std::cerr<< "Catched a runtime exception: " << ex.what() <<std::endl;
            m_Aimms->clearBuffers();
        }

        stop();

        closeDataAndSession();
    }

    /*
    * setValues: Efficient and convenient if the data to assign is dense. The
    * setLabels (for sets) and setValues (for data) methods do allow for dense
    * allocation of an identifier. The data will be added in the natural
    * ordering of the domains, thus combining ease of use with efficiency.
    */
    void denseAssignLabels(int setsize) {

        openSessionAndData();

        start("Assign string labels first to the set using setLabels and then use setValues on the IMultiDimData to assign the values to the parameter");

        m_SubSet->setLabels(s_StringLabels, setsize);
        // UpdateData is not neccesary here because after the setLabels method the set is immediately up-to-date.

        int datasize = setsize * setsize * setsize;
        double* values = new double[datasize];

        // Whether a loop like this is neccesary depends on the source of the values.
        double value = 0;
        int pos = 0;
        for (int i = 0; i < setsize; i += 2) {

            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    values[pos++] = ++value;
                }
            }
            // Now insert a two dimensional slice with zero's (the default value), because this is a dense assign method but the data is not fully dense.
            for (int z = 0; z < setsize * setsize; ++z) {
                values[pos++] = 0;
            }
        }

        m_LargePar->setValues(values, datasize);
        delete [] values;

        m_Aimms->updateData();

        stop();

        closeDataAndSession();
    }

    /*
    * Slices: Efficient if the data to be assigned is sparse and distributed in
    * a way slices can be defined. In this example multiple views are created
    * that are all restricted on the same level on a single element. This
    * reduces the dimension.
    */
    void sliceAssignLabels( int setsize) {

        openSessionAndData();

        start("Assign string labels first to the set using setLabels and then use setValues on IDataView slices of the IMultiDimData to assign the values to the parameter");


        m_SubSet->setLabels(s_StringLabels, setsize);


        // Create a filter to slice the parameter.
        aimms::IFilter* filter = m_LargePar->createFilter();

        int vieweddatasize = setsize * setsize;
        double* values = new double[vieweddatasize];

        double value = 0;
        for (int i = 0; i < setsize; i += 2) {
            // For each position of this dimension to which we will assign data, construct the filter to create a DataView. 
            // Notice the +=2, we only need views on the even positions. 
            filter->restrict(0, s_StringLabels[i]);
            aimms::IDataView* slice = m_LargePar->openView(filter);

            int pos = 0;
            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    values[pos++] = ++value;
                }
            }
            slice->setValues(values, vieweddatasize);
        }

        delete [] values;
        m_Aimms->updateData();

        stop();

        closeDataAndSession();
    }

    /*
    * Filtering by call domain: Efficient if the data is sparse and an
    * appropriate subset is available to filter with.
    */
    void filteredAssignLabels(int setsize) {

        openSessionAndData();

        start("Assign string labels first to the set using setLabels and use setValues on an IDataView filtered by a subset to assign the values on the parameter");

        m_SubSet->setLabels(s_StringLabels, setsize);

        // Open a subset on the domainset to use as a call domain on the level to restrict.
        aimms::ISetData* subsubset = m_Aimms->openSet("SubSubSet");
        aimms::IElement* element = m_SubSet->createElement();

        for (int o = 1; o <= setsize; o += 2) {
            element->setOrdinal(o);
            subsubset->add(element);
        }
        m_Aimms->updateData();

        int vieweddatasize = setsize / 2 * setsize * setsize;
        double* values = new double[vieweddatasize];

        // Create a filter to slice the parameter.
        aimms::IFilter* filter = m_LargePar->createFilter();
        filter->restrict(0, subsubset);
        aimms::IDataView* view = m_LargePar->openView(filter);


        double value = 0;
        int pos = 0;
        for (int i = 0; i < setsize; i += 2) {
            for (int j = 0; j < setsize; ++j) {
                for (int k = 0; k < setsize; ++k) {
                    values[pos++] = ++value;
                }
            }
        }
        view->setValues(values, vieweddatasize);

        delete [] values;
        m_Aimms->updateData();

        stop();

        subsubset->close();
        closeDataAndSession();
    }



private:
    // Helper functions
    void openSessionAndData() {
        m_Aimms = 0;
        m_Aimms = aimms::openSession(m_Location.c_str(), m_Project.c_str());
        if (m_Aimms == 0) {
            return;
        }
        m_SubSet = m_Aimms->openSet("LargeSubSet");
        m_LargePar = m_Aimms->openMultiDim("LargePar1");
    }

    void closeDataAndSession() {
        m_LargePar->close();
        m_SubSet->close();
        m_Aimms->close();
        m_Aimms = 0;
    }

    void start(const char* testDescription) {
        std::cout <<"Start Running " <<  testDescription << "..." << std::endl;
        time(&m_StartTime);
    }

    void stop() {
        time_t stop;
        time(&stop);
        std::cout<< "... " <<" took " << difftime (stop,m_StartTime) << " seconds."  << std::endl << std::endl;  
    }



private : 
    std::string m_Location;
    std::string m_Project;
    aimms::ISession* m_Aimms;
    aimms::ISetData* m_SubSet;
    aimms::IMultiDimData* m_LargePar;
    time_t m_StartTime;
};





int main(int argc, const char* argv[]) 
{
    if (argc != 3) {
        std::cerr << "Invalid number of arguments. usage: <location of AIMMS>  <location of project>" << std::endl; 
        return 1;
    }  
    int setsize = 150;

    Performance test(argv[1], argv[2]);

    try {

        test.convenientAssignLabels(setsize);
        test.efficientAssignLabels(setsize);
        test.denseAssignLabels(setsize);
        test.sliceAssignLabels(setsize);
        test.filteredAssignLabels(setsize);

    } catch (aimms::Exception& ex) {
        std::cerr<< "AIMMS exception:" << ex.what() << std::endl;
    } catch (std::exception& ex){
        std::cerr << "Exception:" << ex.what() << std::endl;
    }
}


