#include "aimms/Include.h"
#include "aimms/AIMMS.h"
#include <assert.h>

// This example demonstrates the custom handling of errors.
// It overrides the default behavior in which aimms::exceptions are thrown in all error cases in two ways:
// The LoggingCallback logs all minor errors, and keeps throwing the severe ones
// The DebuggingCallback logs as well, and asserts where errors can only be expected if the model was called erroneously.

// NB: to run with the DebuggingCallback, change the type of Callback in main

class LoggingCallback : public aimms::ICallBack
{
    virtual void onMessages(std::vector<aimms::AimmsMessage>& messages )
    {
        /* Overridden behavior:
        Only throw if an error is caused by the communication layer or the AIMMS model. 
        Send message to std::cout instead in other cases
        */

        for( size_t m = 0; m < messages.size(); ++m){
            switch(messages[m].code)
            {
            case aimms::AM_Info:
                std::cout<<  "info: "<< messages[m].message << std::endl;
                break;
            case aimms::AM_Warning: 
                std::cout<< "warning: "<< messages[m].message << std::endl;
                break;
            case aimms::AM_InvalidUse: 
                std::cout<<  "invalid use: "<< messages[m].message<< std::endl;
                break;
            case aimms::AM_Runtime: 
                std::cout<< "runtime error: "<< messages[m].message<< std::endl;
                break;
            case aimms::AM_License : 
                throw aimms::LicenseException(messages[m].message.c_str());
            case aimms::AM_Communication: 
                throw aimms::CommunicationException(messages[m].message.c_str());
            case aimms::AM_General:  // AIMMS Model errors
                throw aimms::Exception(messages[m].code, messages[0].message.c_str());
            default: 
                throw aimms::Exception(messages[m].code, messages[0].message.c_str());
            }

            /* 
            The rationale behind this distinction is that in the cases in which now exceptions are thrown, 
            the state of the data is not known. For example: if during the flush of the buffers an error on 
            a specific value is detected by the AIMMS model, it is not known which part of the remaining data 
            is already written to the model. 
            */

        }
    }
};

class DebuggingCallback : public aimms::ICallBack
{
    virtual void onMessages(std::vector<aimms::AimmsMessage>& messages )
    {
        /* Overridden behavior:

        assert if an AM_InvalidUse is encountered, because that is an indication of a bug. 
        For example: adding a string to a numerical parameter gives AM_InvalidUse. 
        */

        for( size_t m = 0; m < messages.size(); ++m){
            switch(messages[m].code)
            {
            case aimms::AM_InvalidUse: 
                std::cout<<  "invalid use: "<< messages[m].message<< std::endl;
                assert(false);
                break;
            default:
                std::cout<<  messages[m].code << ":" << messages[m].message << std::endl;
                break;
            }

        }
    }

};


int main(int argc, const char* argv[]) 
{
    if (argc != 3) {
        std::cerr << "Invalid number of arguments. usage: <location of AIMMS>  <location of project>" << std::endl; 
        return 1;
    }  
    aimms::ISession* session = 0;
    try {
        session = aimms::openSession(argv[1],argv[2]);
    } catch (std::exception & e) {
        std::cerr << e.what();
        return 1;
    }

    LoggingCallback cb;
    //Attach the custom Callback to the session.
    session->setCallBack(&cb);

    //To illustrate the behavior of our callback, make some mistakes:

    //Try to retrieve the set Locations as a multidimensional parameter.
    //This is an InvalidUse error: 
    //  LoggingCallback will not throw
    //  DebuggingCallback will assert here. 
    aimms::IMultiDimData* parameterLocations = session->openMultiDim("Locations");

    if(parameterLocations != 0){
        std::cout << "Expected an InvalidUse error!" << std::endl;;
    }

    aimms::IMultiDimData* parameterSupply = session->openMultiDim("Supply");
    //Do not forget to check the result if the callback does not throw on each error!
    if (parameterSupply == 0){
        std::cout << "Did not expect an error here!" << std::endl;
    } else{
        // Since we did not add any depots, we will get a runtime error here:
        parameterSupply->setValue(aimms::Tuple("London"), 20.0);

        // Using "insert" will resolve this error, since it adds London to the index domain of Supply if not existing. 
        parameterSupply->insert(aimms::Tuple("London"), 20.0);

        // Supply is a numeric parameter: it will not allow strings as value:
        // DebuggingCallback will assert.
        parameterSupply->insert(aimms::Tuple("Paris"), "thirty");
    }

    aimms::IMultiDimData* parameterDemand = session->openMultiDim("Demand");

    if (parameterDemand == 0){
        std::cout << "Did not expect an error here!" << std::endl;
    }  else{
        aimms::ITuple* tup = parameterDemand->createTuple();


        // "London" exists in the set Locations (because we added it to Depots by inserting in Supply),
        // but the index domain of Demand is Customers, which is still empty.
        // The nocheck_ functions are less expensive because they do not check this in advance.
        // Cout will not show an error after these actions:
        tup->getElement(0)->nocheck_setLabel("London");
        parameterDemand->setValue(tup, 90);
        tup->close();

        // But this performance improvement comes with a downside:
        try{
            // The mistake will become apparent when the data is sent to the model. 
            // Then it is undetermined wether the inserted value for "London" in Supply is indeed inserted in Aimms.
            // If Londen is added to the domainset and this set is flushed before the Supply is flushed,
            // the Supply of Londen will be set to its new value without an error.
            // If Londen is not added or Supply is flushed earlier (which may be due to an internal flush),
            // then an error will occur and even other buffered modifications to Supply may not have been inserted in Aimms.
            session->updateData();

        } catch (aimms::Exception & e){
            std::cerr << e.what();
        }

        parameterSupply->close();
        parameterDemand->close();
        session->close();
    }

}