#include "aimms/Include.h"


// This basic example demonstrates the code to 
// - Set up the connection to a model, 
// - Assign parameter data, 
// - Solve the model,
// - Retrieve variables


void setParametersBasic(aimms::ISession* session) 
{
    // Use the insert methods to add labels to the domains sets while adding data.

    // Set supply.
    aimms::IMultiDimData* parameterSupply = session->openMultiDim("Supply");
    parameterSupply->insert(aimms::Tuple("London"),20.0);
    parameterSupply->insert(aimms::Tuple("Paris"),30.0);

    // Close data if it is not longer needed.
    parameterSupply->close();

    // Set demand.
    aimms::IMultiDimData* parameterDemand = session->openMultiDim("Demand"); 
    parameterDemand->insert(aimms::Tuple("Oslo"),10.0);
    parameterDemand->insert(aimms::Tuple("Madrid"),15.0);
    parameterDemand->insert(aimms::Tuple("Berlin"),20.0);
    parameterDemand->close();

    // Use the setValue methods to add data without expanding the domain sets.
    // An aimms::RuntimeException will be thrown if a label does not exist in the index domain.

    // Set transportcosts.
    aimms::IMultiDimData* parameterUnitTransportCost = session->openMultiDim("UnitTransportCost");
    try{
        parameterUnitTransportCost->setValue(aimms::Tuple("London","Oslo"),7.0);
        parameterUnitTransportCost->setValue(aimms::Tuple("London","Madrid"),12.0);
        parameterUnitTransportCost->setValue(aimms::Tuple("London","Berlin"),18.0);
        parameterUnitTransportCost->setValue(aimms::Tuple("Paris","Oslo"),14.0);
        parameterUnitTransportCost->setValue(aimms::Tuple("Paris","Madrid"),23.0);
        parameterUnitTransportCost->setValue(aimms::Tuple("Paris","Berlin"),17.0);
    }catch(aimms::RuntimeException& ex){
        std::cerr << "Catched a runtime exception" << ex.what() << std::endl;
    }
    parameterUnitTransportCost->close();
}


void runMainExecution(aimms::ISession* session)
{
    // Retrieve the procedure.
    aimms::IProcedure* procedure = session->openProcedure("MainExecution");

    // Run the procedure.
    procedure->run();

    // Close it afterwards. 
    procedure->close();
}

void retrieveVariablesBasic(aimms::ISession* session,std::ostream& os) 
{

    // Retrieve and display the scalar variable TotalCost.
    // Scalars have their own interface, and are retrieved with the openScalar method.
    aimms::IScalarData* variableTotalCost = session->openScalar("TotalCost");

    double totalcost = variableTotalCost->asDouble();   
    os << variableTotalCost->getName() << ": " << totalcost << std::endl<< std::endl;
    variableTotalCost->close(); 

    // Retrieve and display the two dimensional variable Transport.
    aimms::IMultiDimData* variableTransport = session->openMultiDim("Transport");

    // To access multidimensional data, use an iterator.
    aimms::IIterator* it = variableTransport->createIterator();

    os << variableTransport->getName() << ":" << std::endl;
    while (it->next()) {  // iterate through all by calling next successively

        // The tuple method gives access to the current position. 
        // The asDouble method returns the value at the current position.
        os << it->tuple() <<" : " << it->asDouble() << std::endl;
    }

    os << std::endl;

    variableTransport->close(); // Closing a multidimensional data closes all its open iterators as well.
}

int main(int argc, const char* argv[]) 
{
    aimms::ISession* session = 0;
    if (argc != 3) {
        std::cerr << "Invalid number of arguments. usage: <location of AIMMS>  <location of project>" << std::endl; 
        return 1;
    }  

    try {
        session = aimms::openSession(argv[1],argv[2]);

        setParametersBasic(session);

        runMainExecution(session);

        retrieveVariablesBasic(session,std::cout);


    } catch (std::exception& e){
        std::cerr << e.what();

        if (session) {
            session->clearBuffers(); // discards all buffered modifications 
            session->close();
        }
        return 1;
    }

    try{
        session->close();
        return 0;
    } catch (std::exception& e){
        std::cerr << e.what();
        return 1;
    }
}