#ifndef _INCLUDE_aimms_ITuple
#define _INCLUDE_aimms_ITuple

#include <string>

namespace aimms {

    class IElement;
    class Tuple;

    /* Object that represents an AIMMS tuple. For a more convenient
       (but less efficient in terms of performance) variant see the
       Tuple class.
       
       Tuples can be created using createTuple on an IMultiDimData,
       IDataView or IIterator.                                      */
    class ITuple {
    private:
        // disallow copy constructor and assign operator
        ITuple( const ITuple& );
        ITuple& operator = ( const ITuple& );   
    protected:
        ITuple() {}
        /* use close */
        virtual ~ITuple() {}
    public:
        /**
         * Returns the IElement on the specified dimension.
         *
         * @param dim The dimension for which to retrieve the IElement.
         * @return The IElement on the specified dimension.
         */
        virtual const IElement* getElement(int dim) const = 0;

        /**
         * Returns the IElement on the specified dimension.
         *
         * @param dim The dimension for which to retrieve the IElement.
         * @return The IElement on the specified dimension.
         */
        virtual IElement* getElement(int dim) = 0;

        /* Returns the IElement reference on the specified dimension. */
        const IElement& operator[](int dim) const { 
            return *getElement(dim);
        }
        /* Returns the IElement reference on the specified dimension. */
        IElement& operator[](int dim){
            return *getElement(dim);
        }


        /**
         * Returns the dimension of the tuple.
         *
         * @return The dimension of the tuple.
         */
        virtual int getDimension() const = 0;

        /**
         * Returns whether all elements in this tuple are valid with respect to
         * their associated sets.
         *
         * @return True if all elements are valid, false otherwise.
         */
        virtual bool isValid() const = 0;

        /**
         * Returns a new ITuple with the same content.
         *
         * @return A new ITuple with the same content.
         */
        virtual ITuple* clone() const = 0;

        /**
         * Closes and invalidates this tuple.
         */
        virtual void close() = 0;

        /**
         * Copies the content of the source ITuple to this ITuple.
         *
         * @param source The ITuple to copy from.
         */
        virtual void copy(const ITuple* tup) = 0;

        /**
         * Assigns a Tuple to this ITuple object
         *
         * @param tup The source Tuple object.
         */
        virtual void setTuple(const Tuple& tup) = 0;
        
        /**
         * Prints the labels of the elements in the tuple to the stream. 
         *
         * @param os The output stream to print to to.
         */
        virtual void printLabels(std::ostream& os) const = 0;
    };

    /* Prints the labels of the elements in the tuple to the stream. */
    inline std::ostream& operator << ( std::ostream& os, const aimms::ITuple* tup )
    {
        tup->printLabels(os);
        return os;
    }
}

#endif
