#ifndef _INCLUDE_aimms_ISet
#define _INCLUDE_aimms_ISet

#include "aimms/IData.h"

namespace aimms{

    class IElement;

    /* Represents the data of a set in the AIMMS model.
       
       Set data objects can be created using openSet on the
       ISession.                                            */                         
    class ISetData : virtual public IData
    {
    private:
        // disallow copy and assign
        ISetData( const ISetData& );
        ISetData& operator = ( const ISetData& );        
    protected:
        ISetData(){}
        /* Use close() to destroy the object. */
        virtual ~ISetData(){}

    public: 
        /**
         * Returns true if this set contains the specified IElement.
         *
         * @param element An IElement of which the presence in this set is to be
         * tested.
         * @return True if this set contains the specified element.
         */
        virtual bool contains(const IElement* element) const = 0;

        /**
         * Returns true if this set contains the specified label. Does not apply to
         * subsets of Integers.
         *
         * @param label A label of which the presence in this set is to be tested.
         * @return True if this set contains the specified label.
         */
        virtual bool contains(const char* label) const = 0;

        /**
         * Returns true if this set contains the specified label. Applies to subsets
         * of Integers.
         *
         * @param label A label of which the presence in this set is to be tested.
         * @return True if this set contains the specified label.
         */
        virtual bool contains(int label) const = 0;

        /**
         * Adds the specified label to this set and all of its supersets. Applies to
         * subsets of Integers.
         *
         * @param label The integer label to be added to this set.
         */
        virtual void add(int label)=0;
        

        /**
         * Adds the specified label to this set and all of its supersets. Does not
         * apply to subsets of Integers.
         *
         * @param label The label to be added to this set.
         */
        virtual void add(const char* label)=0;

        /**
         * Adds the specified element to this set and all of its supersets.
         *
         * @param element The IElement to be added to this set.
         */
        virtual void add(const IElement* element)=0;

        /* Adds the specified labels to this set and all its supersets.
           Does not apply to subsets of Integers.
           
           An exception is thrown if one of the sets in which labels
           must be inserted is read-only.
           
           
           @param labels  The labels to add to this set.                */
        virtual void appendLabels(const char* const labels[], int amount) = 0;

        /**
         * Inserts the specified labels into this set and all its supersets. The set
         * itself and all its subsets are emptied first. Does not apply to subsets
         * of Integers.
         *
         * @param labels The labels to assign to this set.
         */
        virtual void setLabels(const char* const labels[], int amount) = 0;

        /* Adds the specified labels to this set and all its supersets.
           Applies to subsets of Integers.
           
           An exception is thrown if one of the sets in which labels
           must be inserted is read-only.
           
           
           @param labels  The labels to add to this set.                */
        virtual void appendLabels(const int labels[], int amount) = 0;

        /**
         * Inserts the specified labels into this set and all of its supersets. The
         * set itself and all its subsets are emptied first. Applies to subsets of
         * Integers.
         *
         * @param labels The integer labels to assign to this set.
         */
        virtual void setLabels(const int labels[], int amount) = 0;

        /**
         * Removes the specified element from this set and all of its subsets.
         *
         * @param element IElement to be removed from this set.
         */
        virtual void remove(const IElement* element) = 0;

        /**
         * Deletes the specified label from this set and all of its subsets. Only
         * applies to subsets of Integers.
         *
         * @param label The integer label to be removed from this set.
         */
        virtual void remove(int label)=0;

        /**
         * Deletes the specified label from this set and all of its subsets. Does
         * not apply to subsets of Integers.
         *
         * @param label The label to be removed from this set.
         */
        virtual void remove(const char* label)=0;

        /* Creates an ISetIterator to iterate through the data of this
           ISetData.
           
           
           
           Also initializes the iterator: Moves the current position to
           before the beginning of the data; a subsequent next call will
           find the first element.
           <returns>
           An ISetIterator to iterate through the data of this ISetData.
           
           </returns>                                                    */
        virtual class ISetIterator* createIterator() = 0;

        /**
         * Creates an IElement associated with the set of this ISetData.
         *
         * @return An IElement associated with the set of this ISetData.
         */
        virtual IElement* createElement() = 0;


        /**
         * Verifies that the underlying root set of the IElement is applicable on
         * this set. The current value is not considered.
         *
         * @param element The IElement to be tested.
         * @return True if the IElement is compatible with this ISetData.
         */
        virtual bool elementCompatible(const IElement* pos) const = 0 ;
    };
};

#endif // #ifndef _INCLUDE_aimms_ISet
