#ifndef _INCLUDE_aimms_ISession
#define _INCLUDE_aimms_ISession

#include "aimms/ImportExport.h"
#include "aimms/IConfig.h"

#include <vector>

namespace aimms {

    class IData;
    class IProcedure;
    class ICallBack;

    /* Represents a specific AIMMS session associated with a certain
     * model.
     *  
     *  Sessions can be created using the global function
     *  aimms::openSession.                                         
     */
    class ISession {
    private:
        // disallow copy and assign
        ISession( const ISession& );
        ISession& operator = ( const ISession& );   
        
    protected:
        ISession() {};
        /* Use close(). */
        virtual ~ISession() {};
    public:
        /**
         * Flushes all buffers, closes the connection with the model and invalidates
         * all objects still open (elements, tuples, iterators, references to data
         * and procedures).
         */
        virtual void close() = 0;
        /**
         * Flushes all buffers and closes and invalidates all open references to
         * data, tuples, elements and iterators. As opposed to the close method,
         * IProcedures and the ISession itself will remain in a valid state. This
         * method allows a more fine grained control on the memory used by the SDK
         * during a procedure run and the synchronization of the data afterwards.
         */
        virtual void closeAllData() = 0;


        /**
         * Replaces the prior callback with the given ICallback. This can be used to
         * gain more control over the way errors are handled.
         *
         * @param cb The new ICallBack to which errors are dispatched.
         */
        virtual void setCallBack(aimms::ICallBack* cb) = 0;

        /**
         * Opens and creates an IData object that represents the data of the AIMMS
         * identifier with the provided name.
         *
         * @param szName The name of the AIMMS identifier.
         * @return An instance of an IData object.
         */
        virtual IData* openData(const char* szName) = 0;

        /**
         * Opens and creates an ISetData object that represents the data of the
         * AIMMS set identifier with the provided name.
         *
         * @param szName The name of the AIMMS set identifier.
         * @return An instance of an ISetData object.
         */
        virtual class ISetData* openSet(const char* szName) = 0;

        /**
         * Opens and creates an IScalarData object that represents the data of the
         * scalar AIMMS identifier with the provided name.
         *
         * @param szName The name of the scalar AIMMS identifier.
         * @return An instance of an IScalarData object.
         */
        virtual class IScalarData* openScalar(const char* szName) = 0;

        /**
         * Opens and creates an IMultiDimData object that represents the data of the
         * multidimensional AIMMS identifier with the provided name.
         *
         * @param szName The name of the multidimensional AIMMS identifier.
         * @return An instance of an IMultiDimData object.
         */
        virtual class IMultiDimData* openMultiDim(const char* szName) = 0;

        /**
         * Opens and creates an IProcedure object that represents the AIMMS
         * procedure with the provided name.
         *
         * @param szName The name of the procedure.
         * @return An instance of an IProcedure object.
         */
        virtual IProcedure* openProcedure(const char* szName) = 0;

        /**
         * Flushes all outstanding buffers to AIMMS and synchronizes the open SDK
         * objects with the data in AIMMS. If the timeout period is exceeded, a
         * CommunicationException exception is raised, leaving the data in an
         * undefined state.
         *
         * @param timeout Timeout in seconds. A value of -1 is equivalent to infinite.
         */
        virtual void updateData(int timeOut = -1) = 0;

        /**
         * Flushes all outstanding buffers to AIMMS and synchronizes the open SDK
         * objects with the data in AIMMS. If the timeout period is exceeded, a
         * CommunicationException exception is raised, leaving the data in an
         * undefined state.
         *
         * @param changedData Receives a list of IData objects that have been changed
         * since the last call to updateData.
         * @param timeout Timeout in seconds. If the timeout is less than zero, the
         * earlier configured timeout is used. A value of -1 is equivalent to 
         * infinite.
         */
        virtual void updateData(std::vector<class IData*>& changedData, int timeout = -1) = 0;

        /**
         * Empties all buffers, enabling the session to close without further
         * errors.
         */
        virtual void clearBuffers() = 0;

        /**
         * \Returns the configuration settings associated with the ISession.
         *
         *
         * @return The configuration.
         */
        virtual const IConfig* getConfig() = 0;
	};
};

#endif // #ifndef _INCLUDE_aimms_ISession
