#ifndef _INCLUDE_aimms_IScalar
#define _INCLUDE_aimms_IScalar

#include "aimms/IData.h"

namespace aimms{

   /* Represents a scalar value in the AIMMS model. This can be an
      actual scalar parameter/variable or an IDataView with
      dimension zero.
      
      Scalar objects can be created using openScalar on the
      ISession.                                                    */                                                
	class IScalarData : virtual  public IData
	{
    private:
        // disallow copy and assign
        IScalarData( const IScalarData& );
        IScalarData& operator = ( const IScalarData& );   
    protected:
		IScalarData(){}
		/* Use close() to destroy the object */
		virtual	~IScalarData(){}

	public:
        /**
         * Sets the scalar data to the specified value. Applies to double valued
         * scalars only.
         *
         * @param value The value to assign.
         */
		virtual void setValue(double value) = 0;

        /**
         * Sets the scalar data to the specified value. Applies to integer and
         * double valued scalars, and element scalars of which the range is a subset
         * of Integers.
         *
         * @param value The value to assign.
         */
		virtual void setValue(int value) = 0;

        /**
         * Sets the scalar data to the specified value. Applies to element valued
         * scalars only.
         *
         * @param value The value to assign.
         */
		virtual void setValue(const IElement* value) = 0;

        /**
         * Sets the scalar data to the specified value. Applies to string valued
         * scalars, and element scalars of which the range is a not subset of
         * Integers.
         *
         * @param value The value to assign.
         */
        virtual void setValue(const char* value) = 0;

        /**
         * Retrieves the value of the scalar data into the given argument.
         * Applies to double valued data only.
         *
         * @param val The double into which the value of the scalar data should
         * retrieved
         */
        virtual void getValue( double& val) = 0;

        /**
         * Retrieves the value of the scalar data into the given argument.
         * Applies to integer valued data only.
         *
         * @param val The int into which the value of the scalar data should
         * retrieved
         */
		virtual void getValue( int& val) = 0;

        /**
         * Retrieves the value of the scalar data into the given argument. The IElement
         * should first have been created using the createElementValue method.
         * Applies to IElement valued data only.
         *
         * @param val The IElement into which the value of the scalar data should
         * retrieved
         */
		virtual void getValue( IElement* val) = 0;

        /**
         * Retrieves the value of the scalar data into the given argument.
         * Applies to string valued data only.
         *
         * @param val The string into which the value of the scalar data should
         * retrieved
         */
		virtual void getValue( std::string& val) = 0;


        /**
         * Creates an IElement associated with the same set as the range of this
         * scalar element valued data.
         *
         * @return A new instance of IElement.
         */
        virtual IElement* createElementValue() = 0;

        
        /* The value of a scalar can be assigned easily by means of the operator '='. 
        The right hand side must match the value type of the scalar, else an InvalidUseException is thrown.
        */

        /* Applies to double valued scalars.  */
        IScalarData& operator=(const double& newvalue)     { setValue(newvalue);return *this; } 
        /* Applies to integer and double valued data, and element
           valued data of which the range is (a subset of) Integers.    */
        IScalarData& operator=(const int& newvalue)        { setValue(newvalue);return *this; } 
        /* Applies to element valued data. */
        IScalarData& operator=(class IElement* newvalue)   { setValue(newvalue);return *this; } 
        /* Applies to string valued data and element valued data of
           which the range is not (a subset of) Integers.                */
        IScalarData& operator=(const char* newvalue)       { setValue(newvalue);return *this; } 
        

        /**
         * Returns the value of the scalar data. Applies to double valued scalars,
         * and to element valued scalar of which the range is a subset of Integers.
         *
         * @return The integer value.
         */
        virtual const int& asInt() = 0;

        /**
         * Returns the value of the scalar data. Applies to double valued scalars
         * only.
         *
         * @return The double value.
         */
        virtual const double& asDouble() = 0;

        /**
         * Returns the value of the scalar data. Applies to string valued scalars,
         * and to element valued scalar of which the range is not a subset of
         * Integers.
         *
         * @return The string value.
         */
        virtual const std::string& asString() = 0;

        /**
         * Returns the value of the scalar data. Applies to element valued scalars
         * only.
         *
         * @return The read-only IElement value.
         */
        virtual const IElement* asElement() = 0;
    };
};

#endif // #ifndef _INCLUDE_aimms_IScalar
