#ifndef _INCLUDE_aimms_IProcedure
#define _INCLUDE_aimms_IProcedure

#include "aimms/IAimmsObject.h"
#include <vector>

namespace aimms{

	class IData;

    /* Represents a procedure (i.e. not a function) in an AIMMS
       model.
       
       Procedure objects can be created using openProcedure on the
       ISession.                                                   */
	class IProcedure : virtual public IAimmsObject
	{
    private:
        // disallow copy and assign
        IProcedure( const IProcedure& );
        IProcedure& operator = ( const IProcedure& );   
    protected:
		IProcedure(){}
		/* use close */
		virtual ~IProcedure(){}
	public:
        /**
         * Returns the number of procedure arguments.
         *
         * @return The number of procedure arguments.
         */
		virtual int getNumberOfArguments() = 0;

        /**
         * Returns the IData object associated with the given n-th argument.
         *
         * @param index The 0-based index of the argument.
         * @return An IData object associated with the given index-th argument.
         */
		virtual class IData* getArgument(int index) = 0;
        
        /**
         * Runs the procedure with the provided arguments; this methods blocks
         * further execution until the AIMMS procedure has returned or the timeout
         * expires. In the latter case an CommunicationException is raised.
         *
         * @param args The arguments to run the procedure with.
         * @param timeout The amount of time in milliseconds to wait for the procedure to
         * finish. A value of -1 is equivalent to infinite.
         * @return The return value of the AIMMS procedure.
         */
		virtual int run(std::vector<IData*>& args, int timeout = -1) = 0;

        /**
         * Runs the procedure with the provided arguments; this methods blocks
         * further execution until the AIMMS procedure has returned or the timeout
         * expires. In the latter case an CommunicationException is raised.
         *
         * @param arg1 The 1st argument to run the procedure with.
         * @param arg2 The 2nd argument to run the procedure with.
         * @param arg3 The 3rd argument to run the procedure with.
         * @param arg4 The 4th argument to run the procedure with.
         * @param arg5 The 5th argument to run the procedure with.
         * @param arg6 The 6th argument to run the procedure with.
         * @param arg7 The 7th argument to run the procedure with.
         * @param arg8 The 8th argument to run the procedure with.
         * @param timeout The amount of time in milliseconds to wait for the procedure to
         * finish. A value of -1 is equivalent to infinite.
         * @return The return value of the AIMMS procedure.
         */
		virtual int run(IData* arg1=0,IData* arg2=0,IData* arg3=0,IData* arg4=0,
			            IData* arg5=0,IData* arg6=0,IData* arg7=0,IData* arg8=0, int timeout=-1 )=0;

        
        /** 
         * Aborts run. 
         * @param timeout The amount of time in milliseconds to wait for AIMMS to abort the procedure. 
         * A value of -1 is equivalent to infinite.
         * @return False when the procedure was not running, true otherwise.
         */
        virtual bool abort(int timeout = -1) = 0;
	};
};

#endif // #ifndef _INCLUDE_aimms_IProcedure
