#ifndef _INCLUDE_aimms_IIterator
#define _INCLUDE_aimms_IIterator

#include <string>
#include "aimms/ITuple.h"
#include "aimms/IElement.h"

namespace aimms{

    class Tuple;

    /* Reads multi dimensional data in a sequential order.
       
       Iterators can be created using createIterator on an
       IMultiDimData or IDataView. If created on the data itself ,
       an IFilter may be specified.                                */
	class IIterator 
	{
    private:
        // disallow copy and assign
        IIterator( const IIterator& );
        IIterator& operator = ( const IIterator& );   
    protected:
		IIterator(){}
		/* use close */
		virtual ~IIterator(){}

	public:
        /**
         * Closes the iterator and invalidates this object.
         */
        virtual void close() = 0;

        /**
         * Verifies whether the given ITuple can be used with the getTuple method.
         * More specifically, it tests whether the given ITuple is compatible with
         * the domain sets of the associated IData object.
         *
         * @param tuple The ITuple to test.
         * @return True if the tuple is valid, false otherwise.
         */
        virtual bool tupleValid(const ITuple*) = 0;

        /* Re initializes the iterator: Moves the current position to
           before the beginning of the data; a subsequent next call will
           find the first non-default position. If the iterator was in
           an erroneous state, the reset method will also reset the
           state to normal.                                              */
		virtual void reset() = 0;

        /**
         * Moves the current position to the next non-default value, if any.
         *
         * @return False if the end of the data is reached, true otherwise.
         */
		virtual bool next() = 0;

        /**
         * Returns the dimension of the IIterator.
         *
         * @return The dimension of the IIterator.
         */
        virtual int getDimension() = 0;
		
        /**
         * Creates an ITuple that can be used to retrieve the current position, by
         * using it as an argument of the getTuple method.
         *
         * @return A new ITuple.
         */
		virtual ITuple* createTuple() = 0;

        /**
         * Retrieves the current position into an ITuple, which you must create
         * first by invoking the createTuple method.
         *
         * @param tuple An ITuple object that will contain the current position
         * afterwards.
         */
        virtual void getTuple(ITuple* tuple) = 0;

        /**
         * Returns the current position of the iterator. The returned ITuple is
         * readonly.
         *
         * @return The current position of the iterator.
         */
        virtual const ITuple* tuple() const = 0;

        /**
         * Returns the value on the current position. Applies to double valued data
         * only.
         *
         * @return The value on the current position.
         */
        virtual const double& asDouble() = 0;

        /**
         * Returns the value on the current position. Applies to integer valued data
         * and element valued data of which the range is a subset of Integers.
         *
         * @return The value on the current position.
         */
        virtual const int& asInt() = 0;

        /**
         * Returns the value on the current position. Applies to element valued data
         * only.
         *
         * @return The value on the current position.
         */
        virtual const IElement* asElement() = 0;


        /**
         * Returns the value on the current position. Applies to string valued data
         * and element valued data of which the range is not a subset of Integers.
         *
         * @return The value on the current position.
         */
        virtual const std::string& asString() = 0;

        /**
         * Returns the value on the current position. Applies to string valued data
         * and element valued data of which the range is not a subset of Integers.
         *
         * @return The value on the current position.
         */
        virtual const char* asCString() = 0;


        /* \Returns the position on the specified dimension.  */
        const IElement& operator[](int level) const {
            return *(tuple()->getElement(level));
        };

        /**
         * Creates an IElement that can be used to retrieve the value at the
         * current position, by using it as an argument of the getValue method.
         *
         * @return A new IElement.
         */
		virtual IElement* createElementValue() = 0;

        /**
         * Retrieves the value on the current position into the given argument.
         * Applies to double valued data only.
         *
         * @param val The double into which the value on the current position should
         * retrieved
         */
        virtual void getValue( double& val) = 0;

        /**
         * Retrieves the value on the current position into the given argument.
         * Applies to integer valued data only.
         *
         * @param val The int into which the value on the current position should
         * retrieved
         */
		virtual void getValue( int& val) = 0;

        /**
         * Retrieves the value on the current position into the given argument. The IElement
         * should first have been created using the createElementValue method.
         * Applies to IElement valued data only.
         *
         * @param val The IElement into which the value on the current position should
         * retrieved
         */
		virtual void getValue( IElement* val) = 0;

        /**
         * Retrieves the value on the current position into the given argument.
         * Applies to string valued data only.
         *
         * @param val The string into which the value on the current position should
         * retrieved
         */
		virtual void getValue( std::string& val) = 0;
	};
};

#endif // #ifndef _INCLUDE_aimms_IIterator

