#ifndef _INCLUDE_aimms_Filter
#define _INCLUDE_aimms_Filter

#include "aimms/ISetData.h"
#include "aimms/IElement.h"
#include <iosfwd>
#include <sstream>

namespace aimms 
{

    /* A helper class used to construct a view or a filtered
       iterator on multi-dimensional data.
       
       Filters can be created using createFilter on an IMultiDimData. */
    class IFilter {
    private:
        // disallow copy and assign
        IFilter( const IFilter& );
        IFilter& operator = ( const IFilter& );   
    protected:
        IFilter(){};
        /* Use close() to destroy the filter. */
        virtual ~IFilter(){}
    public: 
        /**
         * Closes the filter and its elements and invalidates the object.
         */
        virtual void close() = 0;

        /**
         * Restricts the specified dimension to the given ISetData, i.e. on this
         * dimension, only elements that are contained in the ISetData are included.
         *
         * @param dim The dimension which has to be restricted.
         * @param domain The set to which this dimension has to be restricted.
         */
        virtual void restrict(int dim, const ISetData* domain) = 0;

        /**
         * Restricts the specified dimension to only include the given IElement.
         *
         * @param dim The dimension which has to be restricted.
         * @param element The IElement which value this dimension has to be
         * restricted.
         */
        virtual void restrict(int dim, const IElement* element) = 0;

        /**
         * Restricts the specified dimension to only include the given string label.
         *
         * @param dim The dimension which has to be restricted.
         * @param label The string label to which this dimension has to be
         * restricted.
         */
        virtual void restrict(int dim, const char* label) = 0;

        /**
         * Restricts the specified dimension to only include the given integer
         * label.
         *
         * @param dim The dimension which has to be restricted.
         * @param label The integer element label to which this dimension has to be
         * restricted.
         *
         */
        virtual void restrict(int dim, int label) = 0;

        /**
         * Prints the a textual representation of this IFilter to the
         * specified outputstream 'os'.
         * @return The String representing the IFilter: fixed dimensions are
         * represented by an '@' followed by the element on which the dimension is
         * fixed. Dimensions filtered by a ISetData are represented by the name of
         * the associated set. Dimensions that are not restricted, are represented
         * by an '*'.
         *
         * @param os The outputstream to write the textual representation of this
         * IElement to.
         */
        virtual void print(std::ostream& os) const = 0;

        /**
         * Returns a string representation of the IFilter.
         *
         * @return The String representing the IFilter: fixed dimensions are
         * represented by an '@' followed by the element on which the dimension is
         * fixed. Dimensions filtered by a ISetData are represented by the name of
         * the associated set. Dimensions that are not restricted, are represented
         * by an '*'.
         */
        inline std::string toString() const{
            std::ostringstream ossTmp;
            this->print(ossTmp);
            return ossTmp.str();
        }
    };
}
/** 
* Prints the IFilter to the stream. 
*/
inline std::ostream& operator << ( std::ostream& os, const aimms::IFilter& filter )
{
    filter.print(os);
    return os;
}

#endif // end #ifndef _INCLUDE_aimms_Filter
