#ifndef _INCLUDE_aimms_IElement
#define _INCLUDE_aimms_IElement

#include <iosfwd>
#include <iostream>
#include "aimms/ImportExport.h"

namespace aimms{


    /**
     * Return value for ordinals in erroneous cases. This value might be
     * returned when invoking e.g. the getOrdinal() member-method.
     */
	const int OrdinalUndefined  = -1;

    /* Represents a single element of an AIMMS set.
       
       Elements can be created in several ways:
         * As part of an ITuple (using createTuple on an
           IMultiDimData, an IDataView or an IIterator).
         * Using createElement on an ISetData or ISetIterator.
         * Using createElementValue on an element&#45;valued IMultiDimData,
           IDataView or IIterator.\ \                                       */
	class IElement
	{
    private:
        // disallow copy and assignment
        IElement( const IElement& );
        IElement& operator = ( const IElement& );   

    protected:
		IElement(){}
		// use close()
		virtual ~IElement(){}
	public:
        /**
         * Assigns the specified string label to this IElement. This method does not
         * apply to IElements associated with a subset of Integers. A
         * RuntimeException is raised if the label does not exist in the root set.
         *
         * @param label The string valued label to assign.
         */
		virtual void setLabel(const char* label) = 0;

        /**
         * Assigns the specified integer label to this IElement. This method does
         * only apply to IElements associated with a subset of Integers.
         *
         * @param label The integer valued label to assign.
         */
		virtual void setIntLabel(const int label) = 0;

        /**
         * Assigns the specified position in the set associated with this IElement
         * to this IElement. A RuntimeException is raised if the ordinal position
         * does not exist in the set.
         *
         * @param ord The ordinal to set.
         */
		virtual void setOrdinal(int ord) = 0;

        /**
         * Returns the string label of this IElement. Does only apply to IElements
         * not associated with subsets of Integers.
         *
         * @return The string label.
         */
        virtual const char* getLabel() const = 0;

        /**
         * Returns the integer label of this IElement. Does only apply to IElements
         * associated with subsets of Integers.
         *
         * @return The integer label.
         */
		virtual int getIntLabel() const = 0;

        /**
         * Returns the ordinal position in the associated set of this IElement. In
         * case of an error, the constant OrdinalUndefined is returned.
         *
         * @return The ordinal position. In case of an error, the constant
         * OrdinalUndefined is returned.
         */
		virtual int getOrdinal() const = 0;

        /**
         * Retrieves the string label of this IElement. Does only apply to IElements
         * not associated with subsets of Integers.
         *
         * @param label The string in which to retrieve the label.
         */
        virtual void getLabel(std::string& label) const = 0;

        /**
         * Retrieves the integer label of this IElement. Does only apply to IElements
         * associated with subsets of Integers.
         *
         * @param label The integer in which to retrieve the label.
         */
		virtual void getIntLabel(int& label) const = 0;

        /**
         * Retrieves the ordinal position in the associated set of this IElement. In
         * case of an error, the constant OrdinalUndefined is returned.
         *
         * @return The integer in which to retrieve ordinal position. In case
         * of an error, the constant OrdinalUndefined will be retrieved.
         */
		virtual void getOrdinal(int& ordinal) const = 0;
		
        /**
         * Sets this IElement to the empty Element (i.e. '' in the AIMMS language).
         */
        virtual void setEmpty() = 0;

        /**
         * Returns whether this IElement is set to the empty Element (i.e. '' in the
         * AIMMS language).
         *
         * @return True when this IElement is set to the empty Element (i.e. '' in
         * the AIMMS language). False otherwise.
         */
        virtual bool isEmpty() const = 0;

        /**
         * Attempts to assign the specified string label to this IElement. Returns
         * true if the assignment succeeded, returns false when the integer label
         * does not exist in the associated set. Only applies to IElements not
         * associated with a subset of integers.
         *
         * @param label The string valued label to assign.
         * @return True if the assignment succeeded, false when the integer label
         * does not exist in the associated set.
         */
		virtual bool check_setLabel(const char* label) = 0;

        /**
         * Attempts to assign the specified integer label to this IElement. Returns
         * true if the assignment succeeded, returns false when the integer label
         * does not exist in the associated subset of Integers.
         *
         * @param label The integer valued label to assign.
         * @return True if the assignment succeeded, false when the integer label
         * does not exist in the associated subset of Integers.
         */
		virtual bool check_setIntLabel(const int label) = 0;

        /**
         * Attempts to assign the specified position in the associated set to this
         * IElement. Returns true if the assignment succeeded, returns false when
         * the ordinal position is invalid.
         *
         * @param ord The ordinal to assign.
         * @return False if the ordinal does not exist in the set.
         */
		virtual bool check_setOrdinal(int ord) = 0;
		
        /**
         * Assigns the specified string label to this IElement without checking
         * whether this label is part of the associated set, as opposed to its
         * counterparts setLabel and check_setLabel. This method does raise an
         * RuntimeException if the label does not exist in the root set. This method
         * does not apply to IElements over subsets of Integers.
         *
         * This function may cause an error at a later stage if the label is not in
         * the Set and the element is added as a position or value in some IData.
         *
         * @param label The string label to assign.
         */
		virtual void nocheck_setLabel(const char* label) = 0;

        /**
         * Assigns the specified integer label to this IElement without checking, as
         * opposed to its counterparts setIntLabel and check_setIntLabel.
         *
         * This function may cause an error at a later stage if the label is not in
         * the set and this IElement was used to assign some value into an IData
         * object.
         *
         * @param label The integer label to assign.
         */
		virtual void nocheck_setIntLabel(const int label) = 0;

        /**
         * Adds the specified string label to the set and all its super sets. Does
         * not apply to IElements that are associated with subsets of Integers.
         *
         * @param label The label to add.
         */
		virtual void add(const char* label) = 0;

        /**
         * Adds the specified integer label to the set and all its super sets. Does
         * only apply to IElements associated with subsets of Integers.
         *
         * @param label The label to add.
         */
	    virtual void add(int label) = 0;
				
        /**
         * Copies the content of the source IElement to this IElement.
         *
         * @param source The IElement to copy from.
         *
         */
        virtual void copy( const IElement* ) = 0;

        /**
         * Returns a reference to a new IElement with the same content.
         *
         * @return A reference to a new IElement with the same content.
         */
        virtual IElement* clone() const = 0;

        /**
         * Closes and invalidates this IElement.
         */
		virtual void close() = 0;

		/* Prints the label to the stream.  */
		virtual void printLabel(std::ostream& os) const = 0; 

        /* Sets the integer label. */
        IElement& operator=(const int& label){
            setIntLabel(label);
            return *this; 
        } 
        /* Sets the label. */
        IElement& operator=(const char* label){
            setLabel(label);
            return *this; 
        }

        /* Returns the string label. */
        virtual operator const char* () const = 0;
    
    };
}

/* Prints the label to the stream. */
inline std::ostream& operator << ( std::ostream& os, const aimms::IElement& elem )
{
    elem.printLabel(os);
    return os;
}

#endif
