#ifndef _INCLUDE_aimms_IDataView
#define _INCLUDE_aimms_IDataView

#include "aimms/IData.h"
#include <string>

namespace aimms{

	class ITuple;
    class Tuple;
    class ISetData;
        
   /* A view on multi-dimensional data.
      
      Views can be created using openView, on an IMultiDimData.
      This method requires a filter that can be created using
      createFilter (on the IMultiDimData).                      */
	class IDataView : virtual  public IData
	{
    private:
        // disallow copy and assign
        IDataView( const IDataView& );
        IDataView& operator = ( const IDataView& );
	protected:
        IDataView(){};
		/* Use close() to destroy the object. */
		virtual ~IDataView(){};


	public:
        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array. Applies to double valued data only.
         *
         * @param values The new values of the data, supplied in order.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValues( const double values[], int numberOfValues ) = 0;

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array. Applies to integer, double and (if the range is
         * a subset of Integers) element valued data.
         *
         * @param values The new values of the data, supplied in order.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValues( const int values[], int numberOfValues ) = 0;

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array. Applies to string and (if the range is no subset
         * of Integers) element valued data.
         *
         * @param values The new values of the data, supplied in order.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValues( const char* const values[], int numberOfValues ) = 0;

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array. Applies to string and (if the range is no subset
         * of Integers) element valued data.
         *
         * @param values The new values of the data, supplied in order.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValues( const std::string values[], int numberOfValues ) = 0;  

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array in column major order. Applies to double valued
         * data only.
         *
         * @param values The new values of the data, supplied in column major
         * ordering.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValuesCMO( const double values[], int numberOfValues ) = 0;

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array in column major order. Applies to integer, double
         * and (if the range is a subset of Integers) element valued data.
         *
         * @param values The new values of the data, supplied in column major
         * ordering.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValuesCMO( const int values[], int numberOfValues ) = 0;

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array in column major order. Applies to string and (if
         * the range is no subset of Integers) element valued data.
         *
         * @param values The new values of the data, supplied in in column major
         * ordering.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValuesCMO( const char* const values[], int numberOfValues ) = 0;

        /**
         * Sets the values for the complete domain of the data, by passing all the
         * values as a dense array in column major order. Applies to string and (if
         * the range is no subset of Integers) element valued data.
         *
         * @param values The new values of the data, supplied in in column major
         * ordering.
         * @param numberOfValues The length of the array of values.
         */
        virtual void setValuesCMO( const std::string values[], int numberOfValues ) = 0;  

        /**
         * Returns double the value for the provided ITuple. Applies to double valued
         * data only.
         *
         * @param tuple The ITuple for which to get the double value.
         * @return The double value.
         */
		virtual double getDouble(const ITuple* tuple) = 0;

        /**
         * Returns the double value for the provided Tuple. Applies to double valued
         * data only.
         *
         * @param tuple The Tuple for which to get the double value.
         * @return The double value.
         */
   		virtual double getDouble(const Tuple& tuple) = 0;


        /**
         * Returns the int value for the provided ITuple. Applies to integer valued
         * data and element valued data of which the range is a subset of Integers.
         *
         * @param tuple The ITuple for which to get the int value.
         * @return The integer value.
         */
		virtual int getInt(const ITuple* tuple) = 0;

        /**
         * Returns the int value for the provided Tuple. Applies to integer valued
         * data and element valued data of which the range is a subset of Integers.
         *
         * @param tuple The Tuple for which to get the int value.
         * @return The integer value.
         */
        virtual int getInt(const Tuple& tuple) = 0;

        /**
         * Returns the IElement value for the provided ITuple. Applies to element
         * valued data only.
         *
         * @param tuple The ITuple for which to get the IElement value.
         * @return The read-only IElement value.
         */
		virtual const IElement* getElement(const ITuple* tuple) = 0;

        /**
         * Returns the IElement value for the provided Tuple. Applies to element
         * valued data only.
         *
         * @param tuple The Tuple for which to get the IElement value.
         * @return The read-only IElement value.
         */
        virtual  const IElement* getElement(const Tuple& tuple) = 0;

        /**
         * Returns the String value for the provided ITuple. Applies to string valued
         * data, and element valued data of which the range is not a subset of
         * Integers.
         *
         * @param tuple The ITuple for which to get the String value.
         * @return The String value.
         */
		virtual const char* getString(const ITuple* tuple) = 0;

        /**
         * Returns the String value for the provided Tuple. Applies to string valued
         * data, and element valued data of which the range is not a subset of
         * Integers.
         *
         * @param tuple The Tuple for which to get the String value.
         * @return The String value.
         */
		virtual const char* getString(const Tuple& tuple) = 0;
	
        /**
         * Returns a reference to the index domain on the provided dimension.
         *
         * @param dimension The dimension for which the index domain is requested.
         * @return A reference to the index domain on the provided dimension.
         */
        virtual const ISetData* getIndexDomain(int dimension) = 0;

        /**
        *  Creates an IIterator to iterate over the data. By default,
        *  100 tuples are cached.
        *  
        *  Also initializes the iterator: Moves the current position to
        *  before the beginning of the data; a subsequent next call will
        *  find the first non-default position.
        *  @param tuplesAhead The amount of tuples to cache. 
        *  @return The IIterator. 
        */
        virtual class IIterator* createIterator(int tuplesAhead = 100) = 0;

        /**
         * Creates a new IIterator to iterate over the data. By default, 100 tuples
         * are cached.
         *
         * @return The IIterator.
         */
		virtual ITuple* createTuple() = 0;

        /**
         * Creates an IElement to hold a value. Applies to element valued data only.
         *
         * @return An IElement to hold a value.
         */
        virtual IElement* createElementValue() = 0;
		
        /**
         * Set the provided position to the value. Applies to double valued data.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const ITuple* pos, double value) = 0;

        /**
         * Set the provided position to the value. Applies to double valued data.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const Tuple& pos, double value) = 0;

        /**
         * Set the provided position to the value. Applies to integer and double
         * valued data, and element valued data of which the range is a subset of
         * Integers.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const ITuple* pos, int value) = 0;

        /**
         * Set the provided position to the value. Applies to integer and double
         * valued data, and element valued data of which the range is a subset of
         * Integers.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const Tuple& pos, int value) = 0;

        /**
         * Set the provided position to the value. Applies to element valued data.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const ITuple* pos, const IElement* value) = 0;

        /**
         * Set the provided position to the value. Applies to element valued data.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const Tuple& pos, const IElement* value) = 0;

        /**
         * Set the provided position to the value. Applies to string valued data,
         * and element valued data of which the range is not a subset of Integers.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
		virtual void setValue(const ITuple* pos, const char* value) = 0;

        /**
         * Set the provided position to the value. Applies to string valued data,
         * and element valued data of which the range is not a subset of Integers.
         *
         * @param pos The position to write on.
         * @param value The value to write.
         */
        virtual void setValue(const Tuple& pos, const char* value) = 0;

        /**
         * Verifies that the dimension and underlying root sets of the ITuple are
         * applicable on this IData. Only structure is validated; the values
         * themselves are not considered.
         *
         * @param tuple The tuple to be checked.
         * @return True if the tuple applies.
         */
        virtual bool tupleCompatible(const ITuple* tup) = 0 ;

        /**
         * Verifies that the current values are valid in the domain of the IData.
         *
         * @param tuple The tuple to be checked.
         * @return True if the values are valid.
         */
		virtual bool tupleValid(const ITuple* tup) = 0 ;

        /**
         * Sets the given position to the specified value. If the specified position
         * does not exist yet in the index domain(s), the index domain(s) will be
         * extended. Applies to double valued data.
         *
         * @param pos The position.
         * @param value The value.
         */
		virtual void insert(const Tuple& pos, double value) = 0;

        /**
         * Sets the given position to the specified value. If the specified position
         * does not exist yet in the index domain(s), the index domain(s) will be
         * extended.
         *
         * If the value does not exist in the associated range (when this is a view
         * on an Element Parameter with a subset of Integer as its range), the
         * boolean insertIntoRange can be set to true to indicate that the value
         * should also be inserted into the range.
         *
         * Applies to element valued data of which the range is a subset of
         * Integers.
         *
         * @param pos The position.
         * @param value The value.
         * @param insertIntoRange True if the range may be extended.
         */
		virtual void insert(const Tuple& pos,  int value, bool insertIntoRange = true) = 0;

        /**
         * Sets the given position to the specified value. If the specified position
         * does not exist yet in the index domain(s), the index domain(s) will be
         * extended. Applies to element valued data.
         *
         * @param pos The position.
         * @param value The value.
         */
		virtual void insert(const Tuple& pos,  const IElement* value) = 0;

        /**
         * Sets the given position to the specified value. If the specified position
         * does not exist yet in the index domain(s), the index domain(s) will be
         * extended.
         *
         * If the value does not exist in the associated range (when this is a view
         * on an Element Parameter which does not have a subset of Integer as its
         * range), the boolean insertIntoRange can be set to true to indicate that
         * the value should also be inserted into the range.
         *
         * Applies to element valued data of which the range is not a subset of
         * Integers.
         *
         * @param pos The position.
         * @param value The value.
         * @param insertIntoRange True if the range may be extended.
         */
		virtual void insert(const Tuple& pos, const char* value, bool insertIntoRange = true) = 0;

        /**
         * Flushes the buffered changes to this data; if possible, this will be
         * performed in the background.
         *
         * @return A boolean indicating whether anything has been flushed.
         */
        virtual bool flush() = 0;

	};
};

#endif // #ifndef _INCLUDE_aimms_IDataView
