#ifndef _INCLUDED__rapi4cxx_Options
#define _INCLUDED__rapi4cxx_Options

#include "aimms/ImportExport.h"

namespace aimms {

   /*
    * Object that represents the configuration for the SDK.
    */
    class IConfig 
    {
    public:
        virtual ~IConfig(){};

        /**
         * \Returns the time (in seconds) to wait for calls over the communication
         * layer to return.
         *
         *
         * @return The timeout in seconds.
         */
        virtual int getTimeout() const = 0;

        /**
         * Sets the time (in seconds) to wait for calls over the communication layer
         * to return. The default value is 10 seconds.
         *
         *
         * @param Timeout The Timeout in seconds.
         */
        virtual void setTimeout(int Timeout) = 0;

        /**
         * \Returns the maximum amount of memory in bytes that the buffers of IData
         * instances can reserve before all data is automatically flushed.
         *
         * @return The MaxBufferSize in bytes.
         */
        virtual int getMaxBufferSize() const = 0;

        /**
         * Sets the maximum amount of memory (in bytes) the buffers of IData
         * instances can reserve before all data is automatically flushed. The
         * default is 32 MB.
         *
         *
         * @param MaxBufferSize The MaxBufferSize in bytes.The minimum required is 5
         * KB. The max buffer size cannot be smaller than the max transmit size.
         */
        virtual void setMaxBufferSize(int MaxBufferSize) = 0;

        /**
         * \Returns the maximum number of outstanding flushes in the background.
         *
         * @return The allowed number of simultaneous flushes in the background.
         */
        virtual const int getMaxFlush() const = 0;

        /**
         * Sets the maximum simultaneous amount of flushes. The default is 128. If
         * more identifiers are flushed, the system will wait.
         *
         *
         * @param maxFlush The allowed number of simultaneous flushes in the
         * background; 0 means no flushes in background; negative generates invalid
         * use error.
         */
        virtual void setMaxFlush(int maxFlush) = 0;

        /**
         * \Returns the maximum amount of memory (in bytes) transmitted in a single
         * packet.
         *
         *
         * @return The maximum memory in bytes that will be transmitted in a single
         * packet.
         */
        virtual const int getMaxTransmitSize() const = 0; 

        /**
         * Set the maximum amount of bytes transmitted in a single packet.
         *
         * @param size The maximum memory in bytes transmitted in a single packet.
         * The minimum required is 5 KB. This maximum transmit size cannot be higher
         * than the maximum buffer size (MaxBufferSize). By default, this value is
         * set to 1 MB.
         */
        virtual void setMaxTransmitSize(int size) = 0; 

        /**
         * \Returns the JSON formatted configuration for the communication layer.
         *
         *
         * @return A string containing the communication configuration in JSON
         * format.
         */
        virtual const char* getCommunicationConfig() const = 0;

         /**
         * Sets the JSON formatted configuration for the communication layer.
         *
         * @param CommunicationConfig The json formatted configuration.
         */
        virtual void setCommunicationConfig(const char* CommunicationConfig) = 0;

        /**
         * \Returns the additional AIMMS command line arguments.
         *
         * @return A String containing the additional AIMMS command line \arguments.
         */
        virtual const char* getAimmsCommandLine() const = 0;

        /**
         * Sets the additional AIMMS command line arguments with which to startup the
         * project; see also the AIMMS User's Guide. The AIMMS SDK will always
         * startup the AIMMS host with an equivalent of the --as-server options.
         *
         * By default, no other command line options are used. You should not provide
         * the project name here.
         *
         * @param AimmsCommandLine The AimmsCommandLine.
         */
        virtual void setAimmsCommandLine(const char* AimmsCommandLine) = 0;

        /**
         * \Returns whether the AIMMS SDK will initialize the logging
         *  framework.
         */
        virtual bool getInitLogger() const = 0;

        /**
         * Sets whether the AIMMS SDK should initialize the logging
         *  framework: only disable this when the log4cxx framework is
         *  already initialized prior to initializing the AIMMS SDK.
         *
         * @param InitLogger Boolean indicating whether the logger should be
         * initialized by the AIMMS SDK or that is initialized externally.
         */
        virtual void setInitLogger( bool InitLogger ) = 0;

        /** 
         * \Returns the name of the name of the log file. This option is
         * used only if the LogConfigFile is not set. The default is
         * empty, which implies no logging.
         *
         * @return The name of the log file.
         */
        virtual const char* getLogFileName() const = 0;

        /** Sets the name of the name of the log file. The level of
         * logging is Info. This option is used only if the
         * LogConfigFile is not set.
         * @param LogFileName String containing the name of the log file
         */
        virtual void setLogFileName( const char* LogFileName ) = 0;

        /**
         * \Returns The name of the log4cxx configuration file.
         *
         * @return The name of the log4cxx configuration file.
         */
        virtual const char* getLogConfigFileName() const = 0;

        /**
         * Sets the name of the config file to initialize the logger
         * with. See the Log4cxx documentation on configuration files. 
         * @param LogConfigFileName String containing the name of the configuration 
         * file for the logger
         */
        virtual void setLogConfigFileName( const char* LogConfigFileName ) = 0;

        /**
         * \Returns the location of the configuration file to initialize logging in
         * the AIMMS host process.
         *
         * If the filename is empty there will be no logging. This is the default.
         *
         * @return The HostLoggerConfig.
         */
        virtual const char* getHostLogConfigFileName() const = 0;

        /**
         * Sets the location of the configuration file to initialize the logger of
         * the AIMMS host process. If the filename is empty, there will be no
         * logging. The default is set to no logging. Note: if the session connects
         * to an already running host, this configuration has no effect.
         *
         *
         * @param HostLoggerConfigFileName The name of the configuration file used
         * to initialize the logger of the AIMMS host process.
         */
        virtual void setHostLogConfigFileName( const char* HostLogConfigFileName ) = 0;

        /**
         * \Returns the location of the folder where the AIMMS host executable
         * (aimmssdk_a3host.exe) should reside.
         *
         * @return The location of the folder where the AIMMS host executable can be
         * found.
         */
        virtual const char* getHostFolder() const = 0;

         /**
         * Sets the location of the folder where the AIMMS host executable
         * (aimmssdk_a3host.exe) resides.
         */
        virtual void setHostFolder( const char* hostFolder ) = 0;

     
        /**
         * Returns the time (in seconds) to wait for the connection with the host
         * to be established in Aimms.OpenSession.
         */
        virtual int getConnectTimeout() const = 0;

        /**
         * Sets the time (in seconds) to wait for the connection with the host
         * to be established in Aimms.OpenSession.
         * The default value is 30 seconds.
         *
         * @param connectTimeout The number of seconds before creating
         * the connection with the host times out.
         */
        virtual void setConnectTimeout( int connectTimeout ) = 0;

        /**
         * Returns the character encoding used to interpret the string values in the SDK. 
         *
         * @return The character encoding used to interpret the string values in the SDK. 
         */
        virtual const char* getCharacterEncoding() const = 0;
        
        /* Sets the character encoding used to interpret the string values in
         * the SDK. The default value is UTF-8. Providing an empty string sets the character encoding to the default.
         * 
         * An InvalidUseException is thrown if the provided character
         * set is an unknown encoding, or an encoding with a code unit
         * larger than 8 bits. 
         * 
         * @param characterEncoding The character encoding to use to interpret the string values in the SDK.
         */
        virtual void setCharacterEncoding(const char* characterEncoding) = 0;

        /**
         * Sets the character encoding of the current locale as the character encoding to interpret the string values in the SDK. 
         */
        virtual void setCurrentLocaleCharacterEncoding() = 0;

        /**
        * Returns the command line options for the host. 
        *
        * @return The command line options for the host.
        */
        virtual const char* getHostCommandLine() const = 0;

        /**
        * Sets the command line options for the host. Note: if the session connects
        * to an already running host, this configuration has no effect.
        */
        virtual void setHostCommandLine(const char* val) = 0;

        /**
        * Returns whether after a time out, procedures will wait for the acknowledgment of the cancel 
        * they sent to the running procedure in AIMMS.
        */
        virtual bool getWaitForProcedureCancel() const = 0;

        /** 
        * Sets whether after a time out, procedures will wait for the acknowledgment of the cancel 
        * they sent to the running procedure in AIMMS.
        */
        virtual void setWaitForProcedureCancel(bool val)  = 0;


        /**
         * Clones this IConfig.
         *
         * @return A deep copy of this IConfig object.
         */
        virtual IConfig* clone() const = 0;
    };
};
// end namespace aimms

#endif // end #ifndef _INCLUDED__rapi4cxx_Exception
